#include "stdafx.h"
#include "ttd.h"

#include "window.h"
#include "gui.h"
#include "viewport.h"
#include "gfx.h"
#include "command.h"
#include "vehicle.h"

static uint _cur_train_build_type;
static bool _remove_button_clicked;
static bool _no_sound;
static byte _build_depot_direction;

struct {
	byte orientation;
	byte numtracks;
	byte platlength;
} _railstation;



static void ShowBuildTrainDepotPicker();
static void ShowStationBuilder();

typedef void OnButtonClick(Window *w);

static void GenericPlaceRail(uint tile, int cmd)
{
	int32 res;

	if (!_remove_button_clicked) {
		res = DoCommandByTile(tile, _cur_train_build_type, cmd, 
			DC_MSG(STR_1011_CAN_T_BUILD_RAILROAD_TRACK) | DC_EXEC | DC_AUTO | DC_NO_WATER, CMD_BUILD_SINGLE_RAIL);		
		if (res != CMD_ERROR)
			SndPlayTileFx(0x1E, tile);
		
	} else {
		res = DoCommandByTile(tile, _cur_train_build_type, cmd, 
			DC_MSG(STR_1012_CAN_T_REMOVE_RAILROAD_TRACK) | DC_EXEC | DC_AUTO | DC_NO_WATER, CMD_REMOVE_SINGLE_RAIL);
		if (res != CMD_ERROR)
			SndPlayTileFx(0x1E, tile);
	}
}

void PlaceRail_N(uint tile)
{
	int cmd = _tile_fract_coords.x > _tile_fract_coords.y ? 4 : 5;
	GenericPlaceRail(tile, cmd);
}

void PlaceRail_NE(uint tile)
{
	if (_remove_button_clicked) {
		GenericPlaceRail(tile, 0);
		return;
	}

	VpStartPlaceSizing(tile, 2);
}

void PlaceRail_E(uint tile)
{
	int cmd = _tile_fract_coords.x + _tile_fract_coords.y <= 15 ? 2 : 3;
	GenericPlaceRail(tile, cmd);
}

void PlaceRail_NW(uint tile)
{
	if (_remove_button_clicked) {
		GenericPlaceRail(tile, 1);
		return;
	}

	VpStartPlaceSizing(tile, 1);
}

void PlaceProc_Demolish(uint tile)
{
	int32 ret;

	ret = DoCommandByTile(tile, 0, 0, 
		DC_EXEC | DC_MSG(STR_00B5_CAN_T_CLEAR_THIS_AREA), CMD_LANDSCAPE_CLEAR);

	if (ret == CMD_ERROR)
		return;

	SndPlayTileFx(0x10, tile);
	CreateEffectVehicleAbove(GET_TILE_X(tile)*16 + 8,GET_TILE_Y(tile)*16 + 8, 2, EV_DEMOLISH);
}

static void GenericRaiseLowerLand(uint tile, int mode)
{
	int32 ret;

	tile = TILE_FROM_XY(GET_TILE_X(tile)*16 + _tile_fract_coords.x + 8, GET_TILE_Y(tile)*16 + _tile_fract_coords.y + 8);

	if (mode) {
		ret = DoCommandByTile(tile, 8, (uint32)mode, 
					DC_EXEC | DC_AUTO | DC_MSG(STR_0808_CAN_T_RAISE_LAND_HERE), CMD_TERRAFORM_LAND);
	} else {
		ret = DoCommandByTile(tile, 8, (uint32)mode, 
					 DC_EXEC | DC_AUTO | DC_MSG(STR_0809_CAN_T_LOWER_LAND_HERE), CMD_TERRAFORM_LAND);
	}

	if (ret == CMD_ERROR) {
		SetRedErrorSquare(_terraform_err_tile);
		return;
	}

	SndPlayTileFx(0x1D, tile);
}

void PlaceProc_LowerLand(uint tile)
{
	GenericRaiseLowerLand(tile, 0);
}

void PlaceProc_RaiseLand(uint tile)
{
	GenericRaiseLowerLand(tile, 1);
}

static int16 _place_depot_offs_xy[4] = { -1,	0x100,	1,	-0x100};

static void PlaceExtraDepotRail(uint tile, uint16 extra)
{
	byte b = _map5[tile];

	if (b & 0xC0 || !(b & (extra >> 8)))
		return;

	DoCommandByTile(tile, _cur_train_build_type, extra & 0xFF, 
		DC_MSG(STR_0006) | DC_AUTO | DC_NO_WATER | DC_EXEC, CMD_BUILD_SINGLE_RAIL);
}

static const uint16 _place_depot_extra[12] = {
	0x604,		0x2102,		0x1202,		0x505,
	0x2400,		0x2801,		0x1800,		0x1401,
	0x2203,		0x904,		0x0A05,		0x1103,
};

void PlaceRail_Depot(uint tile)
{
	int32 ret;
	int dir = _build_depot_direction;
	
	ret = DoCommandByTile(tile, _cur_train_build_type, dir,
		DC_MSG(STR_100E_CAN_T_BUILD_TRAIN_DEPOT) | DC_EXEC | DC_AUTO | DC_NO_WATER, CMD_BUILD_TRAIN_DEPOT);

	if (ret == CMD_ERROR)
		return;

	SndPlayTileFx(0x1E, tile);

	ResetObjectToPlace();

	tile += _place_depot_offs_xy[dir];

	if (IS_TILETYPE(tile, MP_RAILWAY)) {
		PlaceExtraDepotRail(tile, _place_depot_extra[dir]);
		PlaceExtraDepotRail(tile, _place_depot_extra[dir + 4]);
		PlaceExtraDepotRail(tile, _place_depot_extra[dir + 8]);
	}
}

void PlaceRail_Station(uint tile)
{
	int32 ret;

	/* pass orientation, numtracks, platlength and railtype */

	ret = DoCommandByTile(tile, _railstation.orientation | (_railstation.numtracks<<8) | (_railstation.platlength<<16),_cur_train_build_type,
			DC_MSG(STR_100F_CAN_T_BUILD_RAILROAD_STATION) | DC_EXEC | DC_AUTO | DC_NO_WATER, CMD_BUILD_RAILROAD_STATION);

	if (ret == CMD_ERROR)
		return;

	SndPlayTileFx(0x1E, tile);
	ResetObjectToPlace();
}

void PlaceRail_Signals(uint tile)
{
	uint trackstat;
	int i;
	int32 ret;

	trackstat = (byte)GetTileTrackStatus(tile,0);

	if ((trackstat & 0x30) == 0x30) {
		trackstat = (_tile_fract_coords.x <= _tile_fract_coords.y) ? 0x20 : 0x10;
	}

	if ((trackstat & 0x0C) == 0x0C) {
		trackstat = (_tile_fract_coords.x + _tile_fract_coords.y <= 15) ? 4 : 8;
	}

	// Lookup the bit index
	i = 0;
	if (trackstat != 0) {	while (!(trackstat & 1)) { i++; trackstat >>= 1; }}

	if (!_remove_button_clicked) {
		ret = DoCommandByTile(tile, i + (_ctrl_pressed ? 8 : 0), 0, 
			DC_EXEC | DC_AUTO | DC_MSG(STR_1010_CAN_T_BUILD_SIGNALS_HERE), CMD_BUILD_SIGNALS);

		if (ret != CMD_ERROR)
			SndPlayTileFx(0x1E, tile);
	} else {
		ret = DoCommandByTile(tile, i, 0,
			DC_EXEC | DC_AUTO | DC_MSG(STR_1013_CAN_T_REMOVE_SIGNALS_FROM), CMD_REMOVE_SIGNALS);

		if (ret != CMD_ERROR)
			SndPlayTileFx(0x1E, tile);
	}
}

void PlaceRail_Bridge(uint tile)
{
	VpStartPlaceSizing(tile, 0);
}

void PlaceRail_Tunnel(uint tile)
{
	int32 ret;
	uint endtile;

	ret = DoCommandByTile(tile, _cur_train_build_type, 0,
		DC_MSG(STR_5016_CAN_T_BUILD_TUNNEL_HERE) | DC_EXEC | DC_AUTO, CMD_BUILD_TUNNEL);

	endtile = _build_tunnel_endtile;

	if (ret == CMD_ERROR) {
		if (endtile != 0) {
			TileIndex old;

			MarkTileDirtyByTile(endtile);
			
			old = _thd.redsq;
			_thd.redsq = endtile;

			if (old != 0) {
				MarkTileDirtyByTile(old);
			}
		}
		return;
	}

	SndPlayTileFx(0x1E, tile);
	ResetObjectToPlace();
}

void PlaceProc_BuyLand(uint tile)
{
	int32 ret;

	ret = DoCommandByTile(tile, 0,0, 
			DC_MSG(STR_5806_CAN_T_PURCHASE_THIS_LAND) | DC_EXEC | DC_AUTO | DC_NO_WATER, CMD_PURCHASE_LAND_AREA);

	if (ret == CMD_ERROR)
		return;

	SndPlayTileFx(0x1E, tile);
}



void BuildRailClick_N(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<2)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(_cur_train_build_type*4 + 0x4EF, 1, w);

	w->click_state |= (1<<2);
	
	_place_proc = PlaceRail_N;
}

void BuildRailClick_NE(Window *w)
{
	if (!_no_sound)
		SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<3)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(_cur_train_build_type*4 + 0x4F0, 1, w);

	w->click_state |= (1<<3);
	_place_proc = PlaceRail_NE;
}

void BuildRailClick_E(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<4)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(_cur_train_build_type*4 + 0x4F1, 1, w);
	w->click_state |= (1<<4);
	_place_proc = PlaceRail_E;
}


void BuildRailClick_NW(Window *w)
{
	if (!_no_sound)
		SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<5)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(_cur_train_build_type*4 + 0x4F2, 1, w);
	w->click_state |= (1<<5);
	_place_proc = PlaceRail_NW;
}

const uint16 _demolish_animcursor[] = {
	ANIM_CURSOR_LINE(0x2C0, 29)
	ANIM_CURSOR_LINE(0x2C1, 29)
	ANIM_CURSOR_LINE(0x2C2, 29)
	ANIM_CURSOR_LINE(0x2C3, 29)
	ANIM_CURSOR_END()
};

void BuildRailClick_Demolish(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<6)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd((uint32)_demolish_animcursor, 1|SOTP_ANIM_CURSOR, w);
	w->click_state |= (1<<6);
	_place_proc = PlaceProc_Demolish;
}


const uint16 _lower_land_animcursor[] = {
	ANIM_CURSOR_LINE(0x2BB, 29)
	ANIM_CURSOR_LINE(0x2BC, 29)
	ANIM_CURSOR_LINE(0x2BD, 98)
	ANIM_CURSOR_END()
};

void BuildRailClick_Lower(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<7)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd((uint32)_lower_land_animcursor, 2|SOTP_ANIM_CURSOR, w);
	
	w->click_state |= (1<<7);
	_place_proc = PlaceProc_LowerLand;
}


const uint16 _raise_land_animcursor[] = {
	ANIM_CURSOR_LINE(0x2B8, 29)
	ANIM_CURSOR_LINE(0x2B9, 29)
	ANIM_CURSOR_LINE(0x2BA, 98)
	ANIM_CURSOR_END()
};

void BuildRailClick_Raise(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<8)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd((uint32)_raise_land_animcursor, 2|SOTP_ANIM_CURSOR, w);

	w->click_state |= (1<<8);
	_place_proc = PlaceProc_RaiseLand;
}

void BuildRailClick_Depot(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<9)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(0x510, 1, w);

	w->click_state |= (1<<9);
	_place_proc = PlaceRail_Depot;

	ShowBuildTrainDepotPicker();
}

void BuildRailClick_Station(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<10)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(0x514, 1, w);

	w->click_state |= (1<<10);
	
	_place_proc = PlaceRail_Station;

	ShowStationBuilder();
}

static const uint16 _build_signals_animcursor[] = {
	ANIM_CURSOR_LINE(0x50C, 148)
	ANIM_CURSOR_LINE(0x50D, 148)
	ANIM_CURSOR_END()
};

void BuildRailClick_Signals(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<11)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd((uint32)_build_signals_animcursor,	1|SOTP_ANIM_CURSOR, w);

	w->click_state |= (1<<11);
	_place_proc = PlaceRail_Signals;
}

void BuildRailClick_Bridge(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<12)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(0xA21, 1, w);

	w->click_state |= (1<<12);
	_place_proc = PlaceRail_Bridge;
}


void BuildRailClick_Tunnel(Window *w)
{
	uint32 image;

	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<13)) {
		ResetObjectToPlace();
		return;
	}

	image = (_cur_train_build_type == 1 ? 0x983 : (_cur_train_build_type == 2 ? 0x984 : 0x982));

	SetObjectToPlaceWnd(image, 3, w);

	w->click_state |= (1<<13);
	_place_proc = PlaceRail_Tunnel;	

	VpStartPreSizing();
}

void BuildRailClick_Remove(Window *w)
{
	if (w->disabled_state & (1<<14))
		return;
	SetWindowDirty(w);
	SndPlayFx(0x13);
	_thd.make_square_red = !!((w->click_state ^= (1 << 14)) & (1<<14));
}

void BuildRailClick_Sign(Window *w)
{
	SndPlayFx(0x13);
	SetWindowDirty(w);

	if (w->click_state & (1<<15)) {
		ResetObjectToPlace();
		return;
	}

	SetObjectToPlaceWnd(0x12B8, 1, w);

	w->click_state |= (1<<15);
	
	_place_proc = PlaceProc_BuyLand;
}

static OnButtonClick * const _build_railroad_button_proc[] = {
	BuildRailClick_N,
	BuildRailClick_NE,
	BuildRailClick_E,
	BuildRailClick_NW,
	BuildRailClick_Demolish,
	BuildRailClick_Lower,
	BuildRailClick_Raise,
	BuildRailClick_Depot,
	BuildRailClick_Station,
	BuildRailClick_Signals,
	BuildRailClick_Bridge,
	BuildRailClick_Tunnel,
	BuildRailClick_Remove,
	BuildRailClick_Sign,
};


static void BuildRailToolbWndProc(Window *w, WindowEvent *e)
{
	switch(e->event) {
	case WE_PAINT:
		w->disabled_state &= ~(1 << 14);
		if (!(w->click_state & ((1<<2)|(1<<3)|(1<<4)|(1<<5)|(1<<11)))) {
			w->disabled_state |= (1 << 14);
			w->click_state &= ~(1<<14);
		}
		DrawWindowWidgets(w);
		break;
	case WE_CLICK: {
		if (e->click.widget >= 2)
			_build_railroad_button_proc[e->click.widget - 2](w);
	} break;

	case WE_KEYPRESS:
		switch(e->keypress.keycode) {
		case '1': BuildRailClick_N(w); break;
		case '2': BuildRailClick_NE(w); break;
		case '3': BuildRailClick_E(w); break;
		case '4': BuildRailClick_NW(w); break;
		default:
			return;
		}
		e->keypress.cont = false;
		break;

	case WE_PLACE_OBJ:
		_remove_button_clicked = (w->click_state & (1 << 14)) != 0;
		_place_proc(e->place.tile);
		return;

	case WE_PLACE_DRAG: {
		static const byte _sel_methods[] = {VPM_X_OR_Y, VPM_FIX_X, VPM_FIX_Y};
		VpSelectTilesWithMethod(e->place.pt.x, e->place.pt.y, _sel_methods[e->place.userdata]);
		return;
	}

	case WE_PLACE_MOUSEUP:
		if (e->click.pt.x != -1) {
			if (e->place.userdata == 0) {
				ShowBuildBridgeWindow(e->place.pt.x, e->place.pt.y, _cur_train_build_type);
			} else {
				DoCommand(_thd.select_to.x, _thd.select_to.y, PACK_POINT(e->place.pt.x,e->place.pt.y),(e->place.userdata << 4) | _cur_train_build_type,
					DC_MSG(STR_1011_CAN_T_BUILD_RAILROAD_TRACK) | DC_EXEC | DC_AUTO | DC_NO_WATER, CMD_BUILD_RAILROAD_TRACK);
			}
		}
		_no_sound = true;
		if (e->place.userdata == 1) {
			BuildRailClick_NW(w);
		} else if (e->place.userdata == 2) {
			BuildRailClick_NE(w);
		}
		_no_sound = false;
		break;

	case WE_ABORT_PLACE_OBJ:
		w->click_state = 0;
		SetWindowDirty(w);

		w = FindWindowById(WC_BUILD_STATION, 0);
		if (w != NULL) WP(w,def_d).close=true;
		w = FindWindowById(WC_BUILD_DEPOT, 0);
		if (w != NULL) WP(w,def_d).close=true;
		break;

	case WE_PLACE_PRESIZE: {
		uint tile = e->place.tile;
		DoCommandByTile(tile, 0, 0, DC_AUTO, CMD_BUILD_TUNNEL);
		VpSetPresizeRange(tile, _build_tunnel_endtile==0?tile:_build_tunnel_endtile);
	} break;
	}
}

static const Widget _build_railroad_widgets[] = {
{   WWT_CLOSEBOX,     7,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,     7,    11,   347,     0,    13, STR_100A_RAILROAD_CONSTRUCTION, STR_018C_WINDOW_TITLE_DRAG_THIS},
{      WWT_PANEL,     7,     0,    21,    14,    35, 0x4E3, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    22,    43,    14,    35, 0x4E4, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    44,    65,    14,    35, 0x4E5, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    66,    87,    14,    35, 0x4E6, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    88,   109,    14,    35, 0x2BF, STR_018D_DEMOLISH_BUILDINGS_ETC},
{      WWT_PANEL,     7,   110,   131,    14,    35, 0x2B7, STR_018E_LOWER_A_CORNER_OF_LAND},
{      WWT_PANEL,     7,   132,   153,    14,    35, 0x2B6, STR_018F_RAISE_A_CORNER_OF_LAND},
{      WWT_PANEL,     7,   154,   175,    14,    35, 0x50E, STR_1019_BUILD_TRAIN_DEPOT_FOR_BUILDING},
{      WWT_PANEL,     7,   176,   217,    14,    35, 0x512, STR_101A_BUILD_RAILROAD_STATION},
{      WWT_PANEL,     7,   218,   239,    14,    35, 0x50B, STR_101B_BUILD_RAILROAD_SIGNALS},
{      WWT_PANEL,     7,   240,   281,    14,    35, 0xA22, STR_101C_BUILD_RAILROAD_BRIDGE},
{      WWT_PANEL,     7,   282,   303,    14,    35, 0x97E, STR_101D_BUILD_RAILROAD_TUNNEL},
{      WWT_PANEL,     7,   304,   325,    14,    35, 0x2CA, STR_101E_TOGGLE_BUILD_REMOVE_FOR},
{      WWT_PANEL,     7,   326,   347,    14,    35, 0x12B7, STR_0329_PURCHASE_LAND_FOR_FUTURE},
{      WWT_LAST},
};

static const WindowDesc _build_railroad_desc = {
	0x124, 0x16, 0x15C, 0x24,
	WC_BUILD_TOOLBAR,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET,
	_build_railroad_widgets,
	BuildRailToolbWndProc
};

static const Widget _build_monorail_widgets[] = {
{   WWT_CLOSEBOX,     7,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,     7,    11,   347,     0,    13, STR_100B_MONORAIL_CONSTRUCTION, STR_018C_WINDOW_TITLE_DRAG_THIS},
{      WWT_PANEL,     7,     0,    21,    14,    35, 0x4E7, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    22,    43,    14,    35, 0x4E8, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    44,    65,    14,    35, 0x4E9, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    66,    87,    14,    35, 0x4EA, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    88,   109,    14,    35, 0x2BF, STR_018D_DEMOLISH_BUILDINGS_ETC},
{      WWT_PANEL,     7,   110,   131,    14,    35, 0x2B7, STR_018E_LOWER_A_CORNER_OF_LAND},
{      WWT_PANEL,     7,   132,   153,    14,    35, 0x2B6, STR_018F_RAISE_A_CORNER_OF_LAND},
{      WWT_PANEL,     7,   154,   175,    14,    35, 0x50E, STR_1019_BUILD_TRAIN_DEPOT_FOR_BUILDING},
{      WWT_PANEL,     7,   176,   217,    14,    35, 0x512, STR_101A_BUILD_RAILROAD_STATION},
{      WWT_PANEL,     7,   218,   239,    14,    35, 0x50B, STR_101B_BUILD_RAILROAD_SIGNALS},
{      WWT_PANEL,     7,   240,   281,    14,    35, 0xA22, STR_101C_BUILD_RAILROAD_BRIDGE},
{      WWT_PANEL,     7,   282,   303,    14,    35, 0x97F, STR_101D_BUILD_RAILROAD_TUNNEL},
{      WWT_PANEL,     7,   304,   325,    14,    35, 0x2CA, STR_101E_TOGGLE_BUILD_REMOVE_FOR},
{      WWT_PANEL,     7,   326,   347,    14,    35, 0x12B7, STR_0329_PURCHASE_LAND_FOR_FUTURE},
{      WWT_LAST},
};

static const WindowDesc _build_monorail_desc = {
	0x124, 0x16, 0x15C, 0x24,
	WC_BUILD_TOOLBAR,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET,
	_build_monorail_widgets,
	BuildRailToolbWndProc
};

static const Widget _build_maglev_widgets[] = {
{   WWT_CLOSEBOX,     7,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,     7,    11,   347,     0,    13, STR_100C_MAGLEV_CONSTRUCTION, STR_018C_WINDOW_TITLE_DRAG_THIS},
{      WWT_PANEL,     7,     0,    21,    14,    35, 0x4EB, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    22,    43,    14,    35, 0x4EC, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    44,    65,    14,    35, 0x4EE, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    66,    87,    14,    35, 0x4ED, STR_1018_BUILD_RAILROAD_TRACK},
{      WWT_PANEL,     7,    88,   109,    14,    35, 0x2BF, STR_018D_DEMOLISH_BUILDINGS_ETC},
{      WWT_PANEL,     7,   110,   131,    14,    35, 0x2B7, STR_018E_LOWER_A_CORNER_OF_LAND},
{      WWT_PANEL,     7,   132,   153,    14,    35, 0x2B6, STR_018F_RAISE_A_CORNER_OF_LAND},
{      WWT_PANEL,     7,   154,   175,    14,    35, 0x50E, STR_1019_BUILD_TRAIN_DEPOT_FOR_BUILDING},
{      WWT_PANEL,     7,   176,   217,    14,    35, 0x512, STR_101A_BUILD_RAILROAD_STATION},
{      WWT_PANEL,     7,   218,   239,    14,    35, 0x50B, STR_101B_BUILD_RAILROAD_SIGNALS},
{      WWT_PANEL,     7,   240,   281,    14,    35, 0xA22, STR_101C_BUILD_RAILROAD_BRIDGE},
{      WWT_PANEL,     7,   282,   303,    14,    35, 0x980, STR_101D_BUILD_RAILROAD_TUNNEL},
{      WWT_PANEL,     7,   304,   325,    14,    35, 0x2CA, STR_101E_TOGGLE_BUILD_REMOVE_FOR},
{      WWT_PANEL,     7,   326,   347,    14,    35, 0x12B7, STR_0329_PURCHASE_LAND_FOR_FUTURE},
{      WWT_LAST},
};

static const WindowDesc _build_maglev_desc = {
	0x124, 0x16, 0x15C, 0x24,
	WC_BUILD_TOOLBAR,0,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET,
	_build_maglev_widgets,
	BuildRailToolbWndProc
};


static const WindowDesc * const _build_rr_desc[] = {
	&_build_railroad_desc,
	&_build_monorail_desc,
	&_build_maglev_desc,
};

void ShowBuildRailToolbar(int index)
{
	assert_array(index, _build_rr_desc);
	
	DeleteWindowById(WC_BUILD_TOOLBAR, 0);
	
	_cur_train_build_type = (byte)index;
	AllocateWindowDesc(_build_rr_desc[index]);
}



static void StationBuildWndProc(Window *w, WindowEvent *e) {
	switch(e->event) {
	case WE_PAINT: {
		uint bits;
		int x,y;

		if (WP(w,def_d).close)
			return;

		bits = (1<<3) << ( _railstation.orientation);
		bits |= (1<<(5-1)) << (_railstation.numtracks);
		bits |= (1<<(12-1)) << (_railstation.platlength);
		bits |= (1<<19) << (_station_show_coverage);
		w->click_state = bits;
		
		x = _railstation.numtracks;
		y = _railstation.platlength;
		if (_railstation.orientation == 0) intswap(x,y);
		SetTileSelectSize(x, y);
		if (_station_show_coverage)
			SetTileSelectBigSize(-4, -4, 8, 8);

		DrawWindowWidgets(w);
		
		StationPickerDrawSprite(39, 42, _cur_train_build_type, 2);
		StationPickerDrawSprite(107, 42, _cur_train_build_type, 3);

		DrawStringCentered(0x4A, 0x0F, STR_3002_ORIENTATION, 0);
		DrawStringCentered(0x4A, 0x4C, STR_3003_NUMBER_OF_TRACKS, 0);
		DrawStringCentered(0x4A, 0x65, STR_3004_PLATFORM_LENGTH, 0);
		DrawStringCentered(0x4A, 0x7E, STR_3066_COVERAGE_AREA_HIGHLIGHT, 0);

		DrawStationCoverageAreaText(2, 151, (uint)-1);
	} break;
		
	case WE_CLICK: {
		switch(e->click.widget) {
		case 0:
			ResetObjectToPlace();
			break;
		case 3:
		case 4:
			_railstation.orientation = e->click.widget - 3;
			SndPlayFx(0x13);
			SetWindowDirty(w);
			break;

		case 5:
		case 6:
		case 7:
		case 8:
		case 9:
		case 10:
		case 11:
			_railstation.numtracks = (e->click.widget - 5) + 1;
			SndPlayFx(0x13);
			SetWindowDirty(w);
			break;

		case 12:
		case 13:
		case 14:
		case 15:
		case 16:
		case 17:
		case 18:
			_railstation.platlength = (e->click.widget - 12) + 1;
			SndPlayFx(0x13);
			SetWindowDirty(w);
			break;

		case 19:
		case 20:
			_station_show_coverage = e->click.widget - 19;
			SndPlayFx(0x13);
			SetWindowDirty(w);
			break;
		}
	} break;
	
	case WE_MOUSELOOP: {
		if (WP(w,def_d).close) {
			DeleteWindow(w);
			return;
		}

		CheckRedrawStationCoverage(w);
		} break;
	}
}

static const Widget _station_builder_widgets[] = {
{   WWT_CLOSEBOX,     7,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,     7,    11,   147,     0,    13, STR_3000_RAIL_STATION_SELECTION, STR_018C_WINDOW_TITLE_DRAG_THIS},
{      WWT_PANEL,     7,     0,   147,    14,   184, 0x0, 0},
{      WWT_PANEL,    14,     7,    72,    26,    73, 0x0, STR_304E_SELECT_RAILROAD_STATION},
{      WWT_PANEL,    14,    75,   140,    26,    73, 0x0, STR_304E_SELECT_RAILROAD_STATION},

{   WWT_CLOSEBOX,    14,    22,    36,    87,    98, STR_00CB_1, STR_304F_SELECT_NUMBER_OF_PLATFORMS},
{   WWT_CLOSEBOX,    14,    37,    51,    87,    98, STR_00CC_2, STR_304F_SELECT_NUMBER_OF_PLATFORMS},
{   WWT_CLOSEBOX,    14,    52,    66,    87,    98, STR_00CD_3, STR_304F_SELECT_NUMBER_OF_PLATFORMS},
{   WWT_CLOSEBOX,    14,    67,    81,    87,    98, STR_00CE_4, STR_304F_SELECT_NUMBER_OF_PLATFORMS},
{   WWT_CLOSEBOX,    14,    82,    96,    87,    98, STR_00CF_5, STR_304F_SELECT_NUMBER_OF_PLATFORMS},
{   WWT_CLOSEBOX,    14,    97,   111,    87,    98, STR_0335_6, STR_304F_SELECT_NUMBER_OF_PLATFORMS},
{   WWT_CLOSEBOX,    14,   112,   126,    87,    98, STR_0336_7, STR_304F_SELECT_NUMBER_OF_PLATFORMS},

{   WWT_CLOSEBOX,    14,    22,    36,   112,   123, STR_00CB_1, STR_3050_SELECT_LENGTH_OF_RAILROAD},
{   WWT_CLOSEBOX,    14,    37,    51,   112,   123, STR_00CC_2, STR_3050_SELECT_LENGTH_OF_RAILROAD},
{   WWT_CLOSEBOX,    14,    52,    66,   112,   123, STR_00CD_3, STR_3050_SELECT_LENGTH_OF_RAILROAD},
{   WWT_CLOSEBOX,    14,    67,    81,   112,   123, STR_00CE_4, STR_3050_SELECT_LENGTH_OF_RAILROAD},
{   WWT_CLOSEBOX,    14,    82,    96,   112,   123, STR_00CF_5, STR_3050_SELECT_LENGTH_OF_RAILROAD},
{   WWT_CLOSEBOX,    14,    97,   111,   112,   123, STR_0335_6, STR_3050_SELECT_LENGTH_OF_RAILROAD},
{   WWT_CLOSEBOX,    14,   112,   126,   112,   123, STR_0336_7, STR_3050_SELECT_LENGTH_OF_RAILROAD},

{   WWT_CLOSEBOX,    14,    14,    73,   137,   148, STR_02DB_OFF, STR_3065_DON_T_HIGHLIGHT_COVERAGE},
{   WWT_CLOSEBOX,    14,    74,   133,   137,   148, STR_02DA_ON, STR_3064_HIGHLIGHT_COVERAGE_AREA},
{      WWT_LAST},
};

static const WindowDesc _station_builder_desc = {
	-1, -1, 0x94, 0xB9,
	WC_BUILD_STATION,WC_BUILD_TOOLBAR,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET,
	_station_builder_widgets,
	StationBuildWndProc
};

static void ShowStationBuilder()
{
	AllocateWindowDesc(&_station_builder_desc);
}

static void BuildTrainDepotWndProc(Window *w, WindowEvent *e)
{
	switch(e->event) {
	case WE_PAINT: {
		int r;

		w->click_state = (1 << 3) << _build_depot_direction;
		DrawWindowWidgets(w);

		r = _cur_train_build_type;
		DrawTrainDepotSprite(70, 17, 0, r);
		DrawTrainDepotSprite(70, 69, 1, r);
		DrawTrainDepotSprite(2, 69, 2, r);
		DrawTrainDepotSprite(2, 17, 3, r);
		break;
		}
	case WE_CLICK: {
		switch(e->click.widget) {
		case 0:
			ResetObjectToPlace();
			break;
		case 3:
		case 4:
		case 5:
		case 6:
			_build_depot_direction = e->click.widget - 3;
			SndPlayFx(0x13);
			SetWindowDirty(w);
			break;
		}
	} break;
	
	case WE_MOUSELOOP:
		if (WP(w,def_d).close)
			DeleteWindow(w);
		return;
	}
}

static const Widget _build_depot_widgets[] = {
{   WWT_CLOSEBOX,     7,     0,    10,     0,    13, STR_00C5, STR_018B_CLOSE_WINDOW},
{    WWT_CAPTION,     7,    11,   139,     0,    13, STR_1014_TRAIN_DEPOT_ORIENTATION,STR_018C_WINDOW_TITLE_DRAG_THIS},
{      WWT_PANEL,     7,     0,   139,    14,   121, 0x0, 0},
{      WWT_PANEL,    14,    71,   136,    17,    66, 0x0, STR_1020_SELECT_RAILROAD_DEPOT_ORIENTATIO},
{      WWT_PANEL,    14,    71,   136,    69,   118, 0x0, STR_1020_SELECT_RAILROAD_DEPOT_ORIENTATIO},
{      WWT_PANEL,    14,     3,    68,    69,   118, 0x0, STR_1020_SELECT_RAILROAD_DEPOT_ORIENTATIO},
{      WWT_PANEL,    14,     3,    68,    17,    66, 0x0, STR_1020_SELECT_RAILROAD_DEPOT_ORIENTATIO},
{      WWT_LAST},
};

static const WindowDesc _build_depot_desc = {
	-1,-1, 0x8C, 0x7A,
	WC_BUILD_DEPOT,WC_BUILD_TOOLBAR,
	WDF_STD_TOOLTIPS | WDF_STD_BTN | WDF_DEF_WIDGET,
	_build_depot_widgets,
	BuildTrainDepotWndProc
};

static void ShowBuildTrainDepotPicker()
{
	AllocateWindowDesc(&_build_depot_desc);
}

void InitializeRailGui()
{
	_build_depot_direction = 3;	
	_railstation.numtracks = 1;
	_railstation.platlength = 1;
}
