/* $Id: minipng.cpp 10512 2007-07-11 20:33:22Z truelight $ */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "crc32.hpp"
#include "minipng.hpp"

assert_compile(sizeof(PNGChunk) == 8);
assert_compile(sizeof(int) == 4);

/* static */ PNGChunk *PNGChunk::Create(unsigned int datalen)
{
	size_t chunk_len = datalen + 12;
	PNGChunk *obj = reinterpret_cast<PNGChunk*>(malloc(chunk_len));
	obj->SetDataLen(datalen);
	return obj;
}

unsigned int PNGChunk::CalcCRC()
{
	/* Do crc over ID and data */
	return EndianFix(CRC32::DoBlock(&IDRaw, GetDataLen() + 4));
}

void PNGTextChunk::GetValue(char *buf)
{
	size_t len = GetValueLen();
	memcpy(buf, GetValuePtr(), len);
	buf[len] = '\0';
}

/* static */ PNGTextChunk *PNGTextChunk::Create(const char *key, const char *value)
{
	size_t keylen = strlen(key);
	size_t vallen = strlen(value);
	size_t datalen = keylen + 1 + vallen;
	size_t size = sizeof(PNGChunk) + sizeof(unsigned int) + datalen;
	PNGTextChunk *obj = reinterpret_cast<PNGTextChunk*>(malloc(size));
	obj->SetDataLen(datalen);
	obj->SetID('tEXt');
	strcpy(obj->GetKeyPtr(), key);
	strcpy(obj->GetValuePtr(), value); // Writes the '\0' in the space for CRC temporarily, fixed on the line
	obj->RecalcCRC();
	return obj;
}

/**
 * Frees all allocated memory. is called before read, and in ~.
 */
void PNGFile::CleanUp()
{
	free(header_chunk);
	header_chunk = NULL;
	free(footer_chunk);
	footer_chunk = NULL;
	data.DeleteAll();
	txtdata.DeleteAll();
	*filename = '\0';
}

/**
 * Adds one property. All old properties with the same key are deleted.
 */
void PNGFile::SetProp(const char *key, const char *val)
{
	for (unsigned int i = txtdata.count; i > 0; i--) {
		if (strcmp(key, txtdata[i - 1]->GetKey()) == 0)
			txtdata.DeleteNo(i - 1);
	}
	txtdata.Add(PNGTextChunk::Create(key, val));
}

/**
 * Returns the key as const char*, and copies value into buffer.
 */
const char *PNGFile::GetProp(unsigned int i, char *valbuf)
{
	txtdata[i]->GetValue(valbuf);
	return txtdata[i]->GetKey();
}

const char correct_png_header[] = "\x89PNG\x0d\x0a\x1a\x0a";

/**
 * Reads the PNG file and stores all data in internal structures for possible later writing
 * any error and Read will abort immediately, returning an error code.
 * @param fn path to png file.
 */
PNGRetVal PNGFile::Read(const char *fn)
{
	CleanUp();
	strcpy(filename, fn);
	FILE *f = fopen(filename, "rb");
	if (f == NULL) return PNG_FileNotFound;
	/* Read the png header and check it */
	if (fread(png_header, 1, 8, f) != 8) { fclose(f); return PNG_NotPNG; }
	if (memcmp(png_header, correct_png_header, 8) != 0) { fclose(f); return PNG_NotPNG; }

	/* Read a chunk at a time until the footer chunk (IEND) was read */
	while (footer_chunk == NULL) {
		unsigned int datalenraw;
		if (fread(&datalenraw, 1, 4, f) != 4) { fclose(f); return PNG_NotPNG; }
		unsigned int datalen = EndianFix(datalenraw);

		PNGChunk *chunk = PNGChunk::Create(datalen);
		if (fread(&chunk->IDRaw, 1, chunk->GetChunkSize() - 4, f) != chunk->GetChunkSize() - 4) { fclose(f); return PNG_NotPNG; }

		if (!chunk->CheckCRC()) { fclose(f); return PNG_NotPNG; }

		switch (chunk->GetID()) {
			case 'IHDR':
				if (header_chunk != NULL) { free(chunk); fclose(f); return PNG_NotPNG; }
				header_chunk = chunk;
				break;
			case 'IEND':
				if (header_chunk == NULL) { free(chunk); fclose(f); return PNG_NotPNG; }
				footer_chunk = chunk;
				break;
			case 'tEXt':
				if (header_chunk == NULL) { free(chunk); fclose(f); return PNG_NotPNG; }
				txtdata.Add(reinterpret_cast<PNGTextChunk*>(chunk));
				break;
			default:
				if (header_chunk == NULL) { free(chunk); fclose(f); return PNG_NotPNG; }
				data.Add(chunk);
				break;
		}
	}

	fclose(f);
	return PNG_OK;
}

static bool WriteChunk(PNGChunk *chunk, FILE *f)
{
	return fwrite(chunk, 1, chunk->GetChunkSize(), f) == chunk->GetChunkSize();
}

PNGRetVal PNGFile::Write()
{
	FILE *f = fopen(filename, "wb");
	if (!f) { return PNG_AccessDenied; }

	if (fwrite(png_header, 1, 8, f) != 8) { fclose(f); return PNG_WriteError; }

	if (!WriteChunk(header_chunk, f)) { fclose(f); return PNG_WriteError; }

	for (unsigned int i = 0; i < txtdata.count; i++)
		if (!WriteChunk(txtdata[i], f)) { fclose(f); return PNG_WriteError; }

	for (unsigned int j = 0; j < data.count; j++)
		if (!WriteChunk(data[j], f)) { fclose(f); return PNG_WriteError; }

	if (!WriteChunk(footer_chunk, f)) { fclose(f); return PNG_WriteError; }

	fclose(f);
	return PNG_OK;
}
