/* Summermute 1.2
 *
 * Namespace: SM_ / sm_
 */

/* Copyright (c) 2009, The Static Void Project
 *
 * Permission to use, copy, modify, and/or distribute this software for 
 * any purpose with or without fee is hereby granted, provided that the 
 * above copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* Summermute is a simple framework for writing rule-based AI.
 *
 * This is not a generic framework, it implements a very specific
 * type of rule-based AI. The method is forward-chaining, conflict
 * resolution is done by choosing the first applicable rule.
 * The Rete algorithm is not used, the framework assumes that
 * each rule which fires changes the working memory, and that
 * the working memory always changes between calls to the function
 * which applies the rules. Make sure that that each rule does
 * indeed change the working memory or the AI will enter an
 * endless loop. The apply rules function only terminates if
 * there are no matching rules left, design your AI acoordingly.
 *
 * Now all of this might sound overly restrictive but it was a
 * conscious design decision. Because of the restrictive nature of
 * the framework it is very easy to understand how any given Summermute
 * AI works and this is the primary purpose of this framework:
 * to provide a clear, logical structure for AI behavior.
 */ 
 

#include "summer.h"

#define FIRST_RULE(r)   (r)
#define LAST_RULE(r,n)  ((r) + ((n)-1))
#define RULE_MATCHED(r) ((*(r))(SM_STATE) == SM_MATCHED)
#define FIRE_RULE(r)    ((void)(*(r))(SM_FIRE))


/* Applies rules
 */
void sm_apply_rules(const SM_RULE * rules, SM_N_RULES n_rules)
{
        const SM_RULE * rule = FIRST_RULE(rules);
        const SM_RULE * const last_rule = LAST_RULE(rules, n_rules);

        for (;;) {

                if (RULE_MATCHED(rule)) {
                        
                        FIRE_RULE(rule);

                        rule = FIRST_RULE(rules);
                        
                        continue;
                }

                if (rule == last_rule) break;

                ++rule;
        }
}

