/************** COPYRIGHT AND CONFIDENTIALITY INFORMATION *********************
**                                                                          **
** Copyright (c) 2013 Technicolor                                           **
** All Rights Reserved                                                      **
**                                                                          **
** This program contains proprietary information which is a trade           **
** secret of TECHNICOLOR and/or its affiliates and also is protected as     **
** an unpublished work under applicable Copyright laws. Recipient is        **
** to retain this program in confidence and is not permitted to use or      **
** make copies thereof other than as permitted in a written agreement       **
** with TECHNICOLOR, UNLESS OTHERWISE EXPRESSLY ALLOWED BY APPLICABLE LAWS. **
**                                                                          **
******************************************************************************/
#include <linux/kernel.h>
#include <linux/slab.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>
#include "bankmgr_proc.h"
#include "bankmgr_p.h"
#include "bankmgr.h"
#include "legacy_upgrade_info.h"
#include <linux/mtd/mtd.h>

/* Names of proc entries */
#define PROC_BANKMGR "banktable"
#define PROC_BANKMGR_BOOTED "booted"
#define PROC_BANKMGR_NOTBOOTED "notbooted"
#define PROC_BANKMGR_ACTIVE "active"
#define PROC_BANKMGR_INACTIVE "inactive"
#define PROC_BANKMGR_BANKTABLE "dumpbanktable"
#define PROC_BANKMGR_ACTIVEVERSION "activeversion"
#define PROC_BANKMGR_PASSIVEVERSION "passiveversion"

#define INFO_BLOCK_ITEM_NAME_VRSS   0x56525353  /* VRSS */

static int bank_booted = BANK_BOOTED;
static int bank_notbooted = BANK_NOTBOOTED;
static int bank_active = BANK_ACTIVE;
static int bank_inactive = BANK_INACTIVE;


// #define BANKTABLE_DEBUG

/* proc root directory */
struct proc_dir_entry *bankmgr_proc_dir = 0;


static int bankmgr_proc_read_criterion(char *page, char **start, off_t offset, int count, int *eof, void *data)
{
	struct bank req_bank;
	int * criterion = data;
	int length = 0;
	short bankid;

	bankid = bankmgr_get_bank_id(BOOTABLE, *criterion);
	if (bankid < 0) goto end;

	bankmgr_get_bank(bankid, &req_bank);
	length = sprintf(page,"%.16s\n", req_bank.name);

end:
	*eof = 1;
	return length;
}

typedef struct {
	unsigned int IB_Size;
	unsigned int IB_ID;
	void * Data;
}T_IB_H;
static int bankmgr_proc_read_version(char *page, char **start, off_t offset, int count, int *eof, void *data)
{
	struct bank req_bank;
	int * criterion = data;
	int length = 0, retlen;
	short bankid;
	struct mtd_info *mtd;
	unsigned long fvp_start, off;
	unsigned char IBMtdRaw[4];
	unsigned int fvp, InfoBlkOff, InfoLength, tmp_count = 0;
	T_IB_H * pIB_H;
	char *IB_buf = NULL;

	bankid = bankmgr_get_bank_id(BOOTABLE, *criterion);
	if (bankid < 0) goto end;

	bankmgr_get_bank(bankid, &req_bank);

	length = sprintf(page,"%s\n", "Unknown");

	mtd = get_mtd_device_nm(req_bank.name);
	fvp_start = ntohl(req_bank.boot_offset) - ntohl(req_bank.bank_offset);

	mtd_read(mtd, fvp_start, 4, &retlen, &fvp);
	if (fvp != 0) {
		length = sprintf(page,"%s\n", "Unknown");
		goto end;
	}

	off = fvp_start + 12;    /* fvp(4B)+fii(8B) */
	mtd_read(mtd, off, 1, &retlen, &IBMtdRaw[0]);
	mtd_read(mtd, (off + 1), 1, &retlen, &IBMtdRaw[1]);
	mtd_read(mtd, (off + 2), 1, &retlen, &IBMtdRaw[2]);
	mtd_read(mtd, (off + 3), 1, &retlen, &IBMtdRaw[3]);
	InfoBlkOff = (unsigned int)((IBMtdRaw[0]<<24) + (IBMtdRaw[1]<<16 )+ (IBMtdRaw[2]<<8) + (IBMtdRaw[3]));
	/* Build doesn't contain infoblock! */
	if(InfoBlkOff == 0xFFFFFFFF)
		goto end;

	off = InfoBlkOff;
	mtd_read(mtd, off, 1, &retlen, &IBMtdRaw[0]);
	mtd_read(mtd, (off + 1), 1, &retlen, &IBMtdRaw[1]);
	mtd_read(mtd, (off + 2), 1, &retlen, &IBMtdRaw[2]);
	mtd_read(mtd, (off + 3), 1, &retlen, &IBMtdRaw[3]);
	InfoLength = (unsigned int)((IBMtdRaw[0]<<24) + (IBMtdRaw[1]<<16 )+ (IBMtdRaw[2]<<8) + (IBMtdRaw[3]));
	if(InfoLength < 12)
		goto end;

	IB_buf = kmalloc(InfoLength - 4, GFP_KERNEL);
	if(IB_buf == NULL)
		goto end;
	mtd_read(mtd, InfoBlkOff + 4, InfoLength - 4, &retlen, IB_buf);
	pIB_H = (T_IB_H *)IB_buf;
	/* The format we look for in the info blocks: <IB_Size><IB_ID><DATA> */
	do {
		if(ntohl(pIB_H->IB_ID) == INFO_BLOCK_ITEM_NAME_VRSS) { 
			/* Found */
			length = sprintf(page,"%.*s\n", ntohl(pIB_H->IB_Size) - 8, &pIB_H->Data);
			break;
		}
		pIB_H = (T_IB_H *)(IB_buf + ntohl(pIB_H->IB_Size));
		tmp_count += ntohl(pIB_H->IB_Size);
	} while(tmp_count < (InfoLength - 4));

	kfree(IB_buf);

end:
	*eof = 1;
	return length;
}

static int bankmgr_proc_write( struct file *filp, const char __user *buff, 
						unsigned long len, void *data )
{
	char *buffer =  NULL;
	short new_bankid;
	int stringlen = len;

	buffer = kmalloc(len+1, GFP_KERNEL);
	if  (buffer == NULL) {
		return -ENOMEM;
	}
	if  (0 != copy_from_user( buffer, buff, len )) {
		kfree(buffer);
		return -EFAULT;
	}

	/* Convert to a normal string */
	buffer[stringlen] = 0;
	while ((stringlen>0) && (buffer[stringlen-1] == '\n' || buffer[stringlen-1] == '\r')) buffer[--stringlen] = 0;

	new_bankid = bankmgr_get_bank_id_by_name(buffer);
	
	if (new_bankid < 0) {
		printk("Invalid bankname: %s\n", buffer);
	}
	else {
		bankmgr_set_active_bank_id(BOOTABLE, new_bankid);
	}
	kfree(buffer);

	return len;
}

static void handle_legacy_upgrade_info(struct proc_dir_entry *pdir)
{
	int ret = 0;
	ret = load_legacy_upgrade_info();
	if (!pdir || ret)
		return;

	publish_legacy_upgrade_info(pdir);
	return;
}

static void remove_legacy_upgrade_info(struct proc_dir_entry *pdir)
{
	unpublish_legacy_upgrade_info(pdir);
	return;
}

#ifdef BANKTABLE_DEBUG
static int bankmgr_proc_read_banktable(char *page, char **start, off_t offset, int count, int *eof, void *data) {
	struct bank_table* bt = load_btab();
	int i;

	if (bt == NULL) {
		printk("Error while loading banktable\n");
		*eof = 1;
		return 0;
	}

	printk(" checksum: %04X\n", (int)ntohl(bt->checksum));
	printk(" version: %d\n", (int)ntohs(bt->version));
	printk(" num_banks: %d\n", (int)ntohs(bt->num_banks));
	printk(" reserved1: %d\n", (int)ntohs(bt->reserved1));
	printk(" reserved2: %d\n", (int)ntohs(bt->reserved2));

	for (i = 0; i < (int)ntohs(bt->num_banks); ++i) {
		struct bank * b = &(bt->banks[i]);
		printk(" bank %d:\n", i);
		printk("  name: %.16s\n", b->name);
		printk("  type: ");
		switch (ntohl(b->type)) {
		case BOOTABLE:
			printk("bootable\n");
			break;
		case USERFS:
			printk("userfs\n");
			break;
		case CUSTOM:
			printk("custom\n");
			break;
		case FEMTO_AP:
			printk("femto_ap\n");
			break;
		default:
			printk("unknown (%d)\n", (int)ntohl(b->type));
			break;

		}
		printk("  size: %08lX\n", ntohl(b->size));
		printk("  bank-offset: %08lX\n", ntohl(b->bank_offset));
		printk("  boot-offset: %08lX\n", ntohl(b->boot_offset));
		printk("  FVP: %04hX\n", ntohs(b->FVP));
		printk("  flags: %04hX\n", ntohs(b->flags));
	}
	free_btab(bt);

	*eof = 1;
	return 0;
}
#endif

int bankmgr_proc_init(void)
{
	struct proc_dir_entry *entry;

	if( !bankmgr_init() ) {
		printk(KERN_ERR "Failed to find rawstorage\n");
		return 0;
	}

	bankmgr_proc_dir = proc_mkdir(PROC_BANKMGR, NULL);
	if (!bankmgr_proc_dir) {
		printk(KERN_ERR "Error: Could not initialize /proc interface\n");
		return -EFAULT;
	}

	entry = create_proc_read_entry(PROC_BANKMGR_BOOTED, 0444, bankmgr_proc_dir, bankmgr_proc_read_criterion, &bank_booted);
	entry = create_proc_read_entry(PROC_BANKMGR_NOTBOOTED, 0444, bankmgr_proc_dir, bankmgr_proc_read_criterion, &bank_notbooted);
	entry = create_proc_read_entry(PROC_BANKMGR_ACTIVE, 0644, bankmgr_proc_dir, bankmgr_proc_read_criterion, &bank_active);
	if (entry) entry->write_proc = bankmgr_proc_write;
	entry = create_proc_read_entry(PROC_BANKMGR_INACTIVE, 0444, bankmgr_proc_dir, bankmgr_proc_read_criterion, &bank_inactive);
#ifdef BANKTABLE_DEBUG
	entry = create_proc_read_entry(PROC_BANKMGR_BANKTABLE, 0444, bankmgr_proc_dir, bankmgr_proc_read_banktable, 0);
#endif
	entry = create_proc_read_entry(PROC_BANKMGR_ACTIVEVERSION, 0444, bankmgr_proc_dir, bankmgr_proc_read_version, &bank_booted);
	entry = create_proc_read_entry(PROC_BANKMGR_PASSIVEVERSION, 0444, bankmgr_proc_dir, bankmgr_proc_read_version, &bank_notbooted);

	handle_legacy_upgrade_info(bankmgr_proc_dir);

	return 0;
}


int bankmgr_proc_cleanup(void)
{
	if ( bankmgr_proc_dir ) {;
		remove_proc_entry(PROC_BANKMGR_BOOTED, bankmgr_proc_dir);
		remove_proc_entry(PROC_BANKMGR_NOTBOOTED, bankmgr_proc_dir);
		remove_proc_entry(PROC_BANKMGR_ACTIVE, bankmgr_proc_dir);
		remove_proc_entry(PROC_BANKMGR_INACTIVE, bankmgr_proc_dir);
#ifdef BANKTABLE_DEBUG
		remove_proc_entry(PROC_BANKMGR_BANKTABLE, bankmgr_proc_dir);
#endif
		remove_proc_entry(PROC_BANKMGR_ACTIVEVERSION, bankmgr_proc_dir);
		remove_proc_entry(PROC_BANKMGR_PASSIVEVERSION, bankmgr_proc_dir);

		remove_legacy_upgrade_info(bankmgr_proc_dir);

		remove_proc_entry(PROC_BANKMGR, NULL);
	}
	bankmgr_fini();
	return 0;
}

