/*
 *  css soundcard
 *  Copyright (c) by Jaroslav Kysela <perex@perex.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/init.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/jiffies.h>
#include <linux/slab.h>
#include <linux/time.h>
#include <linux/wait.h>
#include <linux/math64.h>
#include <linux/moduleparam.h>
#include <sound/core.h>
#include <sound/control.h>
#include <sound/tlv.h>
#include <sound/pcm.h>
#include <sound/rawmidi.h>
#include <sound/info.h>
#include <sound/initval.h>
#include <mach/coma-alsa.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>

//#define pr_debug printk
MODULE_AUTHOR("Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("css soundcard (/dev/null)");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE("{{ALSA,css soundcard}}");

#define MAX_PCM_DEVICES		4
#define MAX_PCM_SUBSTREAMS	4
#define MAX_MIDI_DEVICES	0

/* defaults */
#define MAX_BUFFER_SIZE		(32*1020) /* full divide by 6 */
#define MIN_PERIOD_SIZE		64
#define MAX_PERIOD_SIZE		MAX_BUFFER_SIZE/4
#define USE_FORMATS 		(SNDRV_PCM_FMTBIT_U8 | SNDRV_PCM_FMTBIT_S16_LE)
#define USE_RATE		SNDRV_PCM_RATE_CONTINUOUS | SNDRV_PCM_RATE_8000_48000
#define USE_RATE_MIN		5500
#define USE_RATE_MAX		48000
#define USE_CHANNELS_MIN 	1
#define USE_CHANNELS_MAX 	3 /* 2 amos */
#define USE_PERIODS_MIN 	1
#define USE_PERIODS_MAX 	1024

static int index[SNDRV_CARDS] = {2,[1 ... (SNDRV_CARDS - 1)] = -1};	/* make sure the first card id will be 2 */
static char *id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;	/* ID for this card */
static int enable[SNDRV_CARDS] = {1,[1 ... (SNDRV_CARDS - 1)] = 0};	/* only one card is supported */
static char *model[SNDRV_CARDS] = {[0 ... (SNDRV_CARDS - 1)] = NULL};
static int pcm_devs[SNDRV_CARDS] = {[0 ... (SNDRV_CARDS - 1)] = 3};
static int pcm_substreams_playback[SNDRV_CARDS] = {[0]=1,[1]=1,[2 ... (SNDRV_CARDS - 1)] = 0};
static int pcm_substreams_capture[SNDRV_CARDS] = {[0]=0,[1]=1,[2]=1,[3 ... (SNDRV_CARDS - 1)] = 0};
static char *pcm_devsnames[SNDRV_CARDS] = {[0]="PCMHD",[1]="PCMFD",[2]="TRACE",[3 ... (SNDRV_CARDS - 1)] = ""};

module_param_array(index, int, NULL, 0444);
MODULE_PARM_DESC(index, "Index value for css soundcard.");
module_param_array(id, charp, NULL, 0444);
MODULE_PARM_DESC(id, "ID string for css soundcard.");
module_param_array(enable, bool, NULL, 0444);
MODULE_PARM_DESC(enable, "Enable this css soundcard.");
module_param_array(model, charp, NULL, 0444);
MODULE_PARM_DESC(model, "Soundcard model.");
module_param_array(pcm_devs, int, NULL, 0444);
MODULE_PARM_DESC(pcm_devs, "PCM devices # (0-2) for css driver.");
module_param_array(pcm_substreams_playback, int, NULL, 0444);
MODULE_PARM_DESC(pcm_substreams_playback, "PCM substreams # (1-128) for css driver.");
module_param_array(pcm_substreams_capture, int, NULL, 0444);
MODULE_PARM_DESC(pcm_substreams_capture, "PCM substreams # (1-128) for css driver.");

static struct platform_device *devices[SNDRV_CARDS];

#define MIXER_ADDR_MASTER	0
#define MIXER_ADDR_LINE		1
#define MIXER_ADDR_MIC		2
#define MIXER_ADDR_SYNTH	3
#define MIXER_ADDR_CD		4
#define MIXER_ADDR_LAST		4

struct css_timer_ops {
	int (*create)(struct snd_pcm_substream *);
	void (*free)(struct snd_pcm_substream *);
	int (*prepare)(struct snd_pcm_substream *);
	int (*start)(struct snd_pcm_substream *);
	int (*stop)(struct snd_pcm_substream *);
	snd_pcm_uframes_t (*pointer)(struct snd_pcm_substream *);
};

struct css_model {
	const char *name;
	int (*playback_constraints)(struct snd_pcm_runtime *runtime);
	int (*capture_constraints)(struct snd_pcm_runtime *runtime);
	u64 formats;
	size_t buffer_bytes_max;
	size_t period_bytes_min;
	size_t period_bytes_max;
	unsigned int periods_min;
	unsigned int periods_max;
	unsigned int rates;
	unsigned int rate_min;
	unsigned int rate_max;
	unsigned int channels_min;
	unsigned int channels_max;
};

struct snd_css {
	struct snd_card *card;
	struct css_model *model;
	struct snd_pcm *pcm;
	struct snd_pcm_hardware pcm_hw;
	spinlock_t mixer_lock;
	int mixer_volume[MIXER_ADDR_LAST+1][2];
	int capture_source[MIXER_ADDR_LAST+1][2];
	const struct css_timer_ops *timer_ops;
};




struct css_model *css_models[] = {
	NULL
};

/*
 * coma_alsa interface
 */
 

struct css_coma_alsa_pcm {
	ktime_t base_time;
	ktime_t period_time;
	atomic_t running;
	uint8_t id;
	u32 pos;
	u32 period ;
	struct tasklet_struct tasklet;
	struct snd_pcm_substream *substream;
};

static void css_coma_alsa_pcm_elapsed(long unsigned int priv)
{
	struct css_coma_alsa_pcm *dpcm = (struct css_coma_alsa_pcm *)priv;
	dpcm->pos += dpcm->period;
	if (atomic_read(&dpcm->running))
	{
		/* open this print only if needed */
		//pr_debug("css_coma_alsa_pcm_elapsed\n");
		snd_pcm_period_elapsed(dpcm->substream);
        }
}

static enum coma_alsa_restart css_coma_alsa_callback(unsigned long int data)
{
	struct css_coma_alsa_pcm *dpcm = (struct css_coma_alsa_pcm *)data;
#if 0	
	// use this in case you don't whant the task let'
	if (atomic_read(&(dpcm->running)))
	{
		dpcm->pos += dpcm->period;
		snd_pcm_period_elapsed(dpcm->substream);
       		pr_debug("css_coma_alsa_callback id=%d pos=%d\n",dpcm->id,dpcm->pos);
	        return coma_alsa_RESTART;
        }
        else
 	return coma_alsa_RESTART;
#else
	if (!atomic_read(&dpcm->running))
		return coma_alsa_NORESTART;
	tasklet_schedule(&dpcm->tasklet);
       
	return coma_alsa_RESTART;
#endif
}

static int css_coma_alsa_start(struct snd_pcm_substream *substream)
{
	struct css_coma_alsa_pcm *dpcm = substream->runtime->private_data;
         
        pr_debug("css_coma_alsa_start");
        
        coma_alsa_start(substream->pcm->device,substream->stream );  // SNDRV_PCM_STREAM_PLAYBACK = 0 else 1
	atomic_set(&(dpcm->running), 1);
	return 0;
}

static int css_coma_alsa_stop(struct snd_pcm_substream *substream)
{
	struct css_coma_alsa_pcm *dpcm = substream->runtime->private_data;
        pr_debug("css_coma_alsa_stop");

 	atomic_set(&(dpcm->running), 0);
        coma_alsa_stop(substream->pcm->device,substream->stream);
	return 0;
}

static inline void css_coma_alsa_sync(struct css_coma_alsa_pcm *dpcm)
{
        pr_debug("css_coma_alsa_sync\n");
	tasklet_kill(&dpcm->tasklet);
}

static snd_pcm_uframes_t
css_coma_alsa_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct css_coma_alsa_pcm *dpcm = runtime->private_data;
#if 0   
	// this can be used to read from the memory the poistion of the css pointer
	u32 pos;

	unsigned int size = runtime->buffer_size*2;
 
        pos = *(volatile u32 *)&(runtime->dma_area[size]);
        
        pr_debug("css_coma_alsa_pointer pos =%d adrress = %p\n",
        	(int)bytes_to_frames(substream->runtime, pos),(void *)&(runtime->dma_area[size]));
	return (bytes_to_frames(substream->runtime, pos);
#endif
        return (snd_pcm_uframes_t)(dpcm->pos % runtime->buffer_size);
} 

static int css_coma_alsa_prepare(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct css_coma_alsa_pcm *dpcm = runtime->private_data;
	unsigned int period,period_bytes, rate,format,channels ,direction,size,size_bytes;
        long unsigned int address , physc;
	long sec;
	unsigned long nsecs;
        int ret = 0;
        
        pr_debug("css_coma_alsa_prepare\n");
	period = runtime->period_size;
	period_bytes = frames_to_bytes(runtime, runtime->period_size);
	rate = runtime->rate;
	sec = period / rate;
	period %= rate;
	nsecs = div_u64((u64)period * 1000000000UL + rate - 1, rate);
        dpcm->id = substream->pcm->device;
        format =runtime->sample_bits;
        channels = runtime->channels;
        address = (long unsigned int)runtime->dma_area;
     //   physc = virt_to_phys((void *)address);
    	physc =	runtime->dma_addr;
        size = runtime->buffer_size;
	size_bytes = frames_to_bytes(runtime,runtime->buffer_size);

        // zero the pointer;
        dpcm->pos = 0; 
        dpcm->period = period;

        direction = substream->stream  ; // SNDRV_PCM_STREAM_PLAYBACK = 0 else 1
        
        pr_debug("id = %d rate =%d format=%d period=%d in bytes\n channels=%d addres=0x%p phisc=0x%p size =%d in bytes, direction=%d\n",
                dpcm->id,rate,format,period_bytes,channels,(void *)address,(void *)physc,size_bytes,direction);
        
        if (direction == 0)
            ret = coma_alsa_open(dpcm->id  ,rate,format ,period_bytes ,channels,0,0,(void *)physc,size_bytes+4);
        else
            ret = coma_alsa_open(dpcm->id  ,rate,format ,period_bytes ,channels,(void *)physc,size_bytes+4,0,0);
         
         
        if ( ret < 0 ){

		if ( ret == -2)
			return coma_alsa_reset(substream->pcm->device,substream->stream);	
                pr_err("css_coma_alsa_prepare coma alsa open -failed%d\n",ret);
                return ret;
         }
        
        dpcm->substream = substream;
        coma_alsa_register_elapsed_cb(dpcm->id ,css_coma_alsa_callback,(unsigned long int)dpcm);
        return 0;
}

static int css_coma_alsa_create(struct snd_pcm_substream *substream)
{
	struct css_coma_alsa_pcm *dpcm;
      
        pr_debug("css_coma_alsa_create\n");

	dpcm = kzalloc(sizeof(*dpcm), GFP_KERNEL);
	if (!dpcm)
		return -ENOMEM;
	substream->runtime->private_data = dpcm;
        // init the coma dsp here 
	dpcm->substream = substream;
	atomic_set(&dpcm->running, 0);
	tasklet_init(&dpcm->tasklet, css_coma_alsa_pcm_elapsed,
		     (unsigned long)dpcm);
	return 0;
}

static void css_coma_alsa_free(struct snd_pcm_substream *substream)
{
	struct css_coma_alsa_pcm *dpcm = substream->runtime->private_data;
        pr_debug("css_coma_alsa_free\n");
        coma_alsa_deregister_elapsed_cb(substream->pcm->device);
        coma_alsa_close(substream->pcm->device);
	css_coma_alsa_sync(dpcm);
	kfree(dpcm);
}

static struct css_timer_ops css_coma_alsa_ops = {
	.create =	css_coma_alsa_create,
	.free =		css_coma_alsa_free,
	.prepare =	css_coma_alsa_prepare,
	.start =	css_coma_alsa_start,
	.stop =		css_coma_alsa_stop,
	.pointer =	css_coma_alsa_pointer,
};


static int css_pcm_preallocate_dma_buffer(struct snd_pcm *pcm, int stream)
{
	struct snd_pcm_substream *substream = pcm->streams[stream].substream;
	struct snd_dma_buffer *buf = &substream->dma_buffer;
	size_t size = MAX_BUFFER_SIZE;
	pr_debug(" css_pcm_preallocate_dma_buffer:\n");

	buf->dev.type = SNDRV_DMA_TYPE_DEV;
	buf->dev.dev = pcm->card->dev;
	buf->private_data = NULL;
	buf->area = dma_alloc_coherent(pcm->card->dev, size * 4,
		&buf->addr, GFP_KERNEL);
	if (!buf->area) {
		pr_err("Failed to allocate dma memory - Please increase uncached DMA memory region\n");
		return -ENOMEM;
	}
        
        pr_debug("buf addr =%p buf phisical addr=%p size=%d\n",buf->area,(void *)buf->addr,size);
	
	buf->bytes = size;
	return 0;
}

static void css_pcm_free_dma_buffers(struct snd_pcm *pcm)
{
	struct snd_pcm_substream *substream;
	struct snd_dma_buffer *buf;
	int stream;
	
	pr_debug(" css_pcm_free_dma_buffers:\n");

	for (stream = 0; stream < 2; stream++) {
		substream = pcm->streams[stream].substream;
		if (!substream)
			continue;

		buf = &substream->dma_buffer;
		if (!buf->area)
			continue;
		dma_free_coherent(NULL, buf->bytes, buf->area, 0);
		buf->area = NULL;
	}
}



/*
 * PCM interface
 */

static int css_pcm_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct snd_css *css = snd_pcm_substream_chip(substream);

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
        	pr_debug("css_pcm_trigger start\n");
		return css->timer_ops->start(substream);
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
        	pr_debug("css_pcm_trigger stop\n");
		return css->timer_ops->stop(substream);
	}
	return -EINVAL;
}

static int css_pcm_prepare(struct snd_pcm_substream *substream)
{
	struct snd_css *css = snd_pcm_substream_chip(substream);
        pr_debug("css_pcm_preparer\n");

	return css->timer_ops->prepare(substream);
}

static snd_pcm_uframes_t css_pcm_pointer(struct snd_pcm_substream *substream)
{
	struct snd_css *css = snd_pcm_substream_chip(substream);

	return css->timer_ops->pointer(substream);
}

static struct snd_pcm_hardware css_pcm_hardware = {
	.info =			(SNDRV_PCM_INFO_MMAP |
				 SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_RESUME |
				 SNDRV_PCM_INFO_MMAP_VALID |
				 SNDRV_PCM_INFO_BATCH),
	.formats =		USE_FORMATS,
	.rates =		USE_RATE,
	.rate_min =		USE_RATE_MIN,
	.rate_max =		USE_RATE_MAX,
	.channels_min =		USE_CHANNELS_MIN,
	.channels_max =		USE_CHANNELS_MAX,
	.buffer_bytes_max =	MAX_BUFFER_SIZE,
	.period_bytes_min =	MIN_PERIOD_SIZE,
	.period_bytes_max =	MAX_PERIOD_SIZE,
	.periods_min =		USE_PERIODS_MIN,
	.periods_max =		USE_PERIODS_MAX,
	.fifo_size =		0,
};




static int css_pcm_hw_params(struct snd_pcm_substream *substream,
			       struct snd_pcm_hw_params *hw_params)
{
        struct snd_pcm_runtime *runtime = substream->runtime;

        pr_debug("css_pcm_hw_params\n");

	snd_pcm_set_runtime_buffer(substream, &substream->dma_buffer);	

	runtime->dma_bytes   = params_buffer_bytes(hw_params);
	runtime->buffer_size = params_buffer_size(hw_params);
        return 0;
					
}

static int css_pcm_hw_free(struct snd_pcm_substream *substream)
{
    //  we pre allocate the buffer no need to free here
        return 0;
}

static int css_pcm_open(struct snd_pcm_substream *substream)
{
	struct snd_css *css = snd_pcm_substream_chip(substream);
	struct css_model *model = css->model;
	struct snd_pcm_runtime *runtime = substream->runtime;
	int err;
      
        pr_debug("css_pcm_open\n");
 
        css->timer_ops = &css_coma_alsa_ops;

	err = css->timer_ops->create(substream);
	if (err < 0)
		return err;

	runtime->hw = css->pcm_hw;

	if (model == NULL)
		return 0;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		if (model->playback_constraints)
			err = model->playback_constraints(substream->runtime);
	} else {
		if (model->capture_constraints)
			err = model->capture_constraints(substream->runtime);
	}
	if (err < 0) {
		css->timer_ops->free(substream);
		return err;
	}
	return 0;
}

static int css_pcm_close(struct snd_pcm_substream *substream)
{
 	struct snd_css *css = snd_pcm_substream_chip(substream);
      
        pr_debug("css_pcm_close\n");
	css->timer_ops->free(substream);
	return 0;
}


static struct snd_pcm_ops css_pcm_ops = {
	.open =		css_pcm_open,
	.close =	css_pcm_close,
	.ioctl =	snd_pcm_lib_ioctl,
	.hw_params =	css_pcm_hw_params,
	.hw_free =	css_pcm_hw_free,
	.prepare =	css_pcm_prepare,
	.trigger =	css_pcm_trigger,
	.pointer =	css_pcm_pointer,
	.silence	= NULL,
};

static int __devinit snd_card_css_pcm(struct snd_css *css, int device,
					int substreams_playback,int substreams_capture)
{
	struct snd_pcm *pcm;
	struct snd_pcm_ops *ops;
	int err;

	pr_debug(" snd_card_css_pcm\n");

	err = snd_pcm_new(css->card, pcm_devsnames[device], device,
			       substreams_playback, substreams_capture, &pcm);
	if (err < 0)
	{
		pr_err("snd_card_css_pcm - %d",err);
		return err;
	}
	css->pcm = pcm;
	ops = &css_pcm_ops;
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, ops);
	pcm->private_data = css;
	pcm->info_flags = 0;
	strcpy(pcm->name, "CSS PCM");
	
	if (substreams_playback) {
	pr_debug(" snd_card_css_pcm-allocating playback %d streams\n",substreams_playback);
		err = css_pcm_preallocate_dma_buffer(pcm,
			SNDRV_PCM_STREAM_PLAYBACK);
		if (err)
			goto out;
	}

	if (substreams_capture) {
	pr_debug(" snd_card_css_pcm-allocating captures %d streams\n",substreams_capture);
		err = css_pcm_preallocate_dma_buffer(pcm,
			SNDRV_PCM_STREAM_CAPTURE);
		if (err)
			goto out;
	}
out:
	pr_err("snd_card_css_pcm - %d",err);
	return err;
}

#ifdef SUPPORT_VOLUME_INTERFACE

/*
 * mixer interface
 */

#define css_VOLUME(xname, xindex, addr) \
{ .iface = SNDRV_CTL_ELEM_IFACE_MIXER, \
  .access = SNDRV_CTL_ELEM_ACCESS_READWRITE | SNDRV_CTL_ELEM_ACCESS_TLV_READ, \
  .name = xname, .index = xindex, \
  .info = snd_css_volume_info, \
  .get = snd_css_volume_get, .put = snd_css_volume_put, \
  .private_value = addr, \
  .tlv = { .p = db_scale_css } }

static int snd_css_volume_info(struct snd_kcontrol *kcontrol,
				 struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = -50;
	uinfo->value.integer.max = 100;
	return 0;
}
 
static int snd_css_volume_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_css *css = snd_kcontrol_chip(kcontrol);
	int addr = kcontrol->private_value;

	spin_lock_irq(&css->mixer_lock);
	ucontrol->value.integer.value[0] = css->mixer_volume[addr][0];
	ucontrol->value.integer.value[1] = css->mixer_volume[addr][1];
	spin_unlock_irq(&css->mixer_lock);
	return 0;
}

static int snd_css_volume_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_css *css = snd_kcontrol_chip(kcontrol);
	int change, addr = kcontrol->private_value;
	int left, right;

	left = ucontrol->value.integer.value[0];
	if (left < -50)
		left = -50;
	if (left > 100)
		left = 100;
	right = ucontrol->value.integer.value[1];
	if (right < -50)
		right = -50;
	if (right > 100)
		right = 100;
	spin_lock_irq(&css->mixer_lock);
	change = css->mixer_volume[addr][0] != left ||
	         css->mixer_volume[addr][1] != right;
	css->mixer_volume[addr][0] = left;
	css->mixer_volume[addr][1] = right;
	spin_unlock_irq(&css->mixer_lock);
	return change;
}

static const DECLARE_TLV_DB_SCALE(db_scale_css, -4500, 30, 0);

#define css_CAPSRC(xname, xindex, addr) \
{ .iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname, .index = xindex, \
  .info = snd_css_capsrc_info, \
  .get = snd_css_capsrc_get, .put = snd_css_capsrc_put, \
  .private_value = addr }

#define snd_css_capsrc_info	snd_ctl_boolean_stereo_info
 
static int snd_css_capsrc_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_css *css = snd_kcontrol_chip(kcontrol);
	int addr = kcontrol->private_value;

	spin_lock_irq(&css->mixer_lock);
	ucontrol->value.integer.value[0] = css->capture_source[addr][0];
	ucontrol->value.integer.value[1] = css->capture_source[addr][1];
	spin_unlock_irq(&css->mixer_lock);
	return 0;
}

static int snd_css_capsrc_put(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol)
{
	struct snd_css *css = snd_kcontrol_chip(kcontrol);
	int change, addr = kcontrol->private_value;
	int left, right;

	left = ucontrol->value.integer.value[0] & 1;
	right = ucontrol->value.integer.value[1] & 1;
	spin_lock_irq(&css->mixer_lock);
	change = css->capture_source[addr][0] != left &&
	         css->capture_source[addr][1] != right;
	css->capture_source[addr][0] = left;
	css->capture_source[addr][1] = right;
	spin_unlock_irq(&css->mixer_lock);
	return change;
}

static struct snd_kcontrol_new snd_css_controls[] = {
css_VOLUME("Master Volume", 0, MIXER_ADDR_MASTER),
css_CAPSRC("Master Capture Switch", 0, MIXER_ADDR_MASTER),
css_VOLUME("Synth Volume", 0, MIXER_ADDR_SYNTH),
css_CAPSRC("Synth Capture Switch", 0, MIXER_ADDR_SYNTH),
css_VOLUME("Line Volume", 0, MIXER_ADDR_LINE),
css_CAPSRC("Line Capture Switch", 0, MIXER_ADDR_LINE),
css_VOLUME("Mic Volume", 0, MIXER_ADDR_MIC),
css_CAPSRC("Mic Capture Switch", 0, MIXER_ADDR_MIC),
css_VOLUME("CD Volume", 0, MIXER_ADDR_CD),
css_CAPSRC("CD Capture Switch", 0, MIXER_ADDR_CD)
};

static int __devinit snd_card_css_new_mixer(struct snd_css *css)
{
	struct snd_card *card = css->card;
	unsigned int idx;
	int err;

	spin_lock_init(&css->mixer_lock);
	strcpy(card->mixername, "css Mixer");

	for (idx = 0; idx < ARRAY_SIZE(snd_css_controls); idx++) {
		err = snd_ctl_add(card, snd_ctl_new1(&snd_css_controls[idx], css));
		if (err < 0)
			return err;
	}
	return 0;
}
#endif //SUPPORT_VOLUME_INTERFACE

#if defined(CONFIG_SND_DEBUG) && defined(CONFIG_PROC_FS)
/*
 * proc interface
 */
static void print_formats(struct snd_css *css,
			  struct snd_info_buffer *buffer)
{
	int i;

	for (i = 0; i < SNDRV_PCM_FORMAT_LAST; i++) {
		if (css->pcm_hw.formats & (1ULL << i))
			snd_iprintf(buffer, " %s", snd_pcm_format_name(i));
	}
}

static void print_rates(struct snd_css *css,
			struct snd_info_buffer *buffer)
{
	static int rates[] = {
		5512, 8000, 11025, 16000, 22050, 32000, 44100, 48000,
		64000, 88200, 96000, 176400, 192000,
	};
	int i;

	if (css->pcm_hw.rates & SNDRV_PCM_RATE_CONTINUOUS)
		snd_iprintf(buffer, " continuous");
	if (css->pcm_hw.rates & SNDRV_PCM_RATE_KNOT)
		snd_iprintf(buffer, " knot");
	for (i = 0; i < ARRAY_SIZE(rates); i++)
		if (css->pcm_hw.rates & (1 << i))
			snd_iprintf(buffer, " %d", rates[i]);
}

#define get_css_int_ptr(css, ofs) \
	(unsigned int *)((char *)&((css)->pcm_hw) + (ofs))
#define get_css_ll_ptr(css, ofs) \
	(unsigned long long *)((char *)&((css)->pcm_hw) + (ofs))

struct css_hw_field {
	const char *name;
	const char *format;
	unsigned int offset;
	unsigned int size;
};
#define FIELD_ENTRY(item, fmt) {		   \
	.name = #item,				   \
	.format = fmt,				   \
	.offset = offsetof(struct snd_pcm_hardware, item), \
	.size = sizeof(css_pcm_hardware.item) }

static struct css_hw_field fields[] = {
	FIELD_ENTRY(formats, "%#llx"),
	FIELD_ENTRY(rates, "%#x"),
	FIELD_ENTRY(rate_min, "%d"),
	FIELD_ENTRY(rate_max, "%d"),
	FIELD_ENTRY(channels_min, "%d"),
	FIELD_ENTRY(channels_max, "%d"),
	FIELD_ENTRY(buffer_bytes_max, "%ld"),
	FIELD_ENTRY(period_bytes_min, "%ld"),
	FIELD_ENTRY(period_bytes_max, "%ld"),
	FIELD_ENTRY(periods_min, "%d"),
	FIELD_ENTRY(periods_max, "%d"),
};

static void css_proc_read(struct snd_info_entry *entry,
			    struct snd_info_buffer *buffer)
{
	struct snd_css *css = entry->private_data;
	int i;

	for (i = 0; i < ARRAY_SIZE(fields); i++) {
		snd_iprintf(buffer, "%s ", fields[i].name);
		if (fields[i].size == sizeof(int))
			snd_iprintf(buffer, fields[i].format,
				*get_css_int_ptr(css, fields[i].offset));
		else
			snd_iprintf(buffer, fields[i].format,
				*get_css_ll_ptr(css, fields[i].offset));
		if (!strcmp(fields[i].name, "formats"))
			print_formats(css, buffer);
		else if (!strcmp(fields[i].name, "rates"))
			print_rates(css, buffer);
		snd_iprintf(buffer, "\n");
	}
}

static void css_proc_write(struct snd_info_entry *entry,
			     struct snd_info_buffer *buffer)
{
	struct snd_css *css = entry->private_data;
	char line[64];

	while (!snd_info_get_line(buffer, line, sizeof(line))) {
		char item[20];
		const char *ptr;
		unsigned long long val;
		int i;

		ptr = snd_info_get_str(item, line, sizeof(item));
		for (i = 0; i < ARRAY_SIZE(fields); i++) {
			if (!strcmp(item, fields[i].name))
				break;
		}
		if (i >= ARRAY_SIZE(fields))
			continue;
		snd_info_get_str(item, ptr, sizeof(item));
		if (strict_strtoull(item, 0, &val))
			continue;
		if (fields[i].size == sizeof(int))
			*get_css_int_ptr(css, fields[i].offset) = val;
		else
			*get_css_ll_ptr(css, fields[i].offset) = val;
	}
}

static void __devinit css_proc_init(struct snd_css *chip)
{
	struct snd_info_entry *entry;

	if (!snd_card_proc_new(chip->card, "css_pcm", &entry)) {
		snd_info_set_text_ops(entry, chip, css_proc_read);
		entry->c.text.write = css_proc_write;
		entry->mode |= S_IWUSR;
		entry->private_data = chip;
	}
}
#else
#define css_proc_init(x)
#endif /* CONFIG_SND_DEBUG && CONFIG_PROC_FS */

static int __devinit snd_css_probe(struct platform_device *devptr)
{
	struct snd_card *card;
	struct snd_css *css;
	int idx, err;
	int dev = devptr->id;
	
	printk("snd_css_probe: dev= %d id =%s pcm_devices=%d \n",dev,id[dev],pcm_devs[dev]);
	
	err = snd_card_create(index[dev], id[dev], THIS_MODULE,
			      sizeof(struct snd_css), &card);
	if (err < 0)
		return err;

	css = card->private_data;
	css->card = card;

	for (idx = 0; idx < MAX_PCM_DEVICES && idx < pcm_devs[dev]; idx++) {
		if (pcm_substreams_playback[idx] + pcm_substreams_capture[idx] <1 )
			continue;
		err = snd_card_css_pcm(css, idx, pcm_substreams_playback[idx],pcm_substreams_capture[idx]);
		if (err < 0)
			goto __nodev;
	}

	css->pcm_hw = css_pcm_hardware;

#ifdef SUPPORT_VOLUME_INTERFACE

	err = snd_card_css_new_mixer(css);
	if (err < 0)
		goto __nodev;
#endif
	strcpy(card->driver, "css");
	strcpy(card->shortname, "css");
	sprintf(card->longname, "css %i", dev + 1);

	css_proc_init(css);

	snd_card_set_dev(card, &devptr->dev);

	err = snd_card_register(card);
	if (err == 0) {
		platform_set_drvdata(devptr, card);
		return 0;
	}
      __nodev:
	snd_card_free(card);
	return err;
}

static int __devexit snd_css_remove(struct platform_device *devptr)
{
 	struct snd_card *card = platform_get_drvdata(devptr);

	struct snd_css *css = card->private_data;
        struct snd_pcm *pcm = css->pcm;
        int stream;
 
        if ( pcm)
        {
 	        for (stream = 0; stream < 2; stream++) 
  
			css_pcm_free_dma_buffers(pcm->streams[stream].pcm);
        }
	snd_card_free(platform_get_drvdata(devptr));
	platform_set_drvdata(devptr, NULL);
	return 0;
}

#ifdef CONFIG_PM
static int snd_css_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct snd_card *card = platform_get_drvdata(pdev);
	struct snd_css *css = card->private_data;

	snd_power_change_state(card, SNDRV_CTL_POWER_D3hot);
	snd_pcm_suspend_all(css->pcm);
	return 0;
}
	
static int snd_css_resume(struct platform_device *pdev)
{
	struct snd_card *card = platform_get_drvdata(pdev);

	snd_power_change_state(card, SNDRV_CTL_POWER_D0);
	return 0;
}
#endif

#define SND_css_DRIVER	"snd_css"

static struct platform_driver snd_css_driver = {
	.probe		= snd_css_probe,
	.remove		= __devexit_p(snd_css_remove),
#ifdef CONFIG_PM
	.suspend	= snd_css_suspend,
	.resume		= snd_css_resume,
#endif
	.driver		= {
		.name	= SND_css_DRIVER
	},
};

static void snd_css_unregister_all(void)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(devices); ++i)
		platform_device_unregister(devices[i]);
	platform_driver_unregister(&snd_css_driver);
//	free_fake_buffer();
}

static int __init alsa_card_css_init(void)
{
	int i, cards, err;

	err = platform_driver_register(&snd_css_driver);
	if (err < 0)
		return err;

	cards = 0;
	for (i = 0; i < SNDRV_CARDS; i++) {
		struct platform_device *device;
		if (! enable[i])
			continue;
		device = platform_device_register_simple(SND_css_DRIVER,
							 i, NULL, 0);
		if (IS_ERR(device))
			continue;
		if (!platform_get_drvdata(device)) {
			platform_device_unregister(device);
			continue;
		}
		devices[i] = device;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		printk(KERN_ERR "css soundcard not found or device busy\n");
#endif
		snd_css_unregister_all();
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_css_exit(void)
{
	snd_css_unregister_all();
}

module_init(alsa_card_css_init)
module_exit(alsa_card_css_exit)
