#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <asm/gpio.h>
#include <linux/earlysuspend.h>
#include <linux/semaphore.h>

#include <video/dw74fb.h>
#include "ili9325.h"

#define LCDC_GPIO_RESET GPIO_PORTA(28)

struct ili9325_plat_data {

#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend ili9325_early_suspend;
#endif
	struct dw74fb *dw74fb;
	char* name;
};


static void ili9325_regindex(struct dw74fb *dw74fb, uint16_t reg)
{
	/* RS=1, LCMD = NOT LAST:0 ,RWCMD:WR-1 */
	dw74fb_writel(dw74fb, LCDC_REG_CMDAR, ILI9325_SET_REGISTER_WRITE);
	dw74fb_writel(dw74fb, LCDC_REG_TXDR,  reg);
	/* Total 2 clk_hclk+3 clk_lcdc required for FIFO trans*/
}

static void ili9325_write_regvalue(struct dw74fb *dw74fb, uint16_t value)
{
	/* RS=0, LCMD = NOT LAST:0 ,RWCMD:WR-1 */
	dw74fb_writel(dw74fb,LCDC_REG_CMDAR, ILI9325_REGISTER_WRITE);
	dw74fb_writel(dw74fb,LCDC_REG_TXDR,  value);
	/* Total 2 clk_hclk+3 clk_lcdc required for FIFO trans */
}

static unsigned long ili9325_read_regvalue(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_CMDAR, ILI9325_REGISTER_READ);
	return (0xFFFF & dw74fb_readl(dw74fb, LCDC_REG_RXDR));
	/* Total 2 clk_hclk+3 clk_lcdc required for FIFO trans*/
}

static void ili9325_write_reg(struct dw74fb *dw74fb,int reg,int value)
{
	/* Set the index register */
	ili9325_regindex(dw74fb, reg);
	/* Send the value to be written to register */
	ili9325_write_regvalue(dw74fb, value);
}

static unsigned long ili9325_read_reg(struct dw74fb *dw74fb,int reg)
{
	/* Set the index register */
	ili9325_regindex(dw74fb, reg);
	/* Send the value to be written to register */
	return ili9325_read_regvalue(dw74fb);
}

static void ili9325_reset(void)
{
	// set the gpio for reseting ili9325=>write 1
	gpio_set_value(LCDC_GPIO_RESET,1);
	msleep(1);

	// set the gpio to get ili9325 from reset==> write 0
	gpio_set_value(LCDC_GPIO_RESET,0);
	msleep(10);

	// set the gpio ili9325 write 1
	gpio_set_value(LCDC_GPIO_RESET,1);
	msleep(50);
	return;
}


#ifdef CONFIG_HAS_EARLYSUSPEND

static void ili9325_early_suspend(struct early_suspend *es) {

	struct ili9325_plat_data* pdata = container_of(es, struct ili9325_plat_data, ili9325_early_suspend);
	struct dw74fb *dw74fb = pdata->dw74fb;

	lcdc_own_hw();
    
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL1, 0x0131); // Set D1=0, D0=1
	msleep(10);
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL1, 0x0130); // Set D1=0, D0=0
	msleep(10);
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL1, 0x0000); // display OFF
	//************* Power OFF sequence **************//
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL1, 0x0000); // SAP, BT[3:0], APE, AP, DSTB, SLP
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL2, 0x0000); // DC1[2:0], DC0[2:0], VC[2:0]
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL3, 0x0000); // VREG1OUT voltage
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL4, 0x0000); // VDV[4:0] for VCOM amplitude
	msleep(200); // Dis-charge capacitor power voltage
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL1, 0x0002); // SAP, BT[3:0], APE, AP, DSTB, SLP

	lcdc_release_hw();
}

static void ili9325_late_resume(struct early_suspend *es) {

	struct ili9325_plat_data* pdata = container_of(es, struct ili9325_plat_data, ili9325_early_suspend);
	struct dw74fb *dw74fb = pdata->dw74fb;

	lcdc_own_hw();

		/* Power on Seq */
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL1, 0x0000); // SAP, BT[3:0], AP, DSTB, SLP
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL2, 0x0000); // DC1[2:0], DC0[2:0], VC[2:0]
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL3, 0x0000); // VREG1OUT voltage
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL4, 0x0000); // VDV[4:0] for VCOM amplitude
	msleep(200); // Dis-charge capacitor power voltage
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL1, 0x1290); // SAP, BT[3:0], AP, DSTB, SLP, STB
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL2, 0x0227); // DC1[2:0], DC0[2:0], VC[2:0]
	msleep(50); // Delay 50ms
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL3, 0x001B); //Inernal reference voltage =Vci;
	msleep(50); // Delay 50ms
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL4, 0x1700); // VDV[4:0] for VCOM amplitude
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL7, 0x001E); // VCM[5:0] for VCOMH
	msleep(50); // Delay 50ms
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL1, 0x0133); // 262K color and display ON

	lcdc_release_hw();
}

#endif 

int ili9325_init(struct dw74fb *dw74fb)
{
	int ChipId;
	
	/* Reset ili9325 */
	ili9325_reset();

	ChipId = ili9325_read_reg(dw74fb,ILI9325_DRIVEROP_CTRL);
	if(ChipId == 0x9325) {
		printk(" Device Communication is success \n");
	} else {
		printk("Error Chip Id = %x ",ChipId);
	}

	/* Intial sequense */
	ili9325_write_reg(dw74fb,ILI9325_DRIVEROP_CTRL, 0x0100); // set SS and SM bit
	ili9325_write_reg(dw74fb,ILI9325_DRV_WAVE_CTRL, 0x0700); // set 1 line inversion

	/* Currently programming for 16 Bit Interface 64K colour, Need to be upgraded
	 * to satisfy 258K Colour that means TRI = 1, DFM = 0*/
	ili9325_write_reg(dw74fb,ILI9325_ENTRY_MODE, 0x1030); // set GRAM write direction and BGR=1.
	ili9325_write_reg(dw74fb,ILI9325_RESIZE_CTRL, 0x0000); // Resize register

	/* Need to check this value */
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL2, 0x0207); // set the back porch and front porch

	/* Need to be modified to reduce scan cycle 
	 * by the way reduce the power consumption */
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL3, 0x0000); // set non-display area refresh cycle ISC[3:0]
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL4, 0x0000); // FMARK function

	/* This needs to be set for 16-Bit , Later this needs to change for 18-Bit */
	ili9325_write_reg(dw74fb,ILI9325_RGB_DISP_CTRL1, 0x0001); // RGB interface setting

	ili9325_write_reg(dw74fb,ILI9325_FRAME_MARKER_POS,0x0000); // Frame marker Position
	ili9325_write_reg(dw74fb,ILI9325_RGB_DISP_CTRL2, 0x0000); // RGB interface polarity
  
	/* Power on Seq */
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL1, 0x0000); // SAP, BT[3:0], AP, DSTB, SLP, STB
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL2, 0x0007); // DC1[2:0], DC0[2:0], VC[2:0]
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL3, 0x0000); // VREG1OUT voltage
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL4, 0x0000); // VDV[4:0] for VCOM amplitude
	msleep(200); // Dis-charge capacitor power voltage

	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL1, 0x1290); // SAP, BT[3:0], AP, DSTB, SLP, STB
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL2, 0x0227); // DC1[2:0], DC0[2:0], VC[2:0]
	msleep(50); // Delay 50ms
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL3, 0x001B); // Internal reference voltage= Vci;
	msleep(50); // Delay 50ms
	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL4, 0x1700); // Set VDV[4:0] for VCOM amplitude


	ili9325_write_reg(dw74fb,ILI9325_POWER_CTRL7, 0x001E); // Set VCM[5:0] for VCOMH
	ili9325_write_reg(dw74fb,ILI9325_FRAME_RATE_CLR_CTRL, 0x000D); // Set Frame Rate
	msleep(50); // Delay 50ms

	ili9325_write_reg(dw74fb,ILI9325_HZ_GRAM_SET, 0x0000); // GRAM horizontal Address
	ili9325_write_reg(dw74fb,ILI9325_VER_GRAM_SET, 0x0000); // GRAM Vertical Address

	/* Set the GAMA Curve */
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL1, 0x0004);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL2, 0x0007);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL3, 0x0006);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL4, 0x0206);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL5, 0x0408);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL6, 0x0507);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL7, 0x0200);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL8, 0x0707);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL9, 0x0504);
	ili9325_write_reg(dw74fb,ILI9325_GAMA_CTRL10, 0x0F02);

	/* Set GRAM Area */
	/* HZ 0->239, VER-0->319 */
	ili9325_write_reg(dw74fb,ILI9325_HZ_ADDR_START, 0x0000); // Horizontal GRAM Start Address
	ili9325_write_reg(dw74fb,ILI9325_HZ_ADDR_END, 0x00EF); // Horizontal GRAM End Address
	ili9325_write_reg(dw74fb,ILI9325_VER_ADDR_START, 0x0000); // Vertical GRAM Start Address
	ili9325_write_reg(dw74fb,ILI9325_VER_ADDR_END, 0x013F); // Vertical GRAM Start Address

	ili9325_write_reg(dw74fb,ILI9325_DRIVER_OP_CTRL, 0xA700); // Gate Scan Line
	ili9325_write_reg(dw74fb,ILI9325_BASE_IMG_DISP, 0x0001); // NDL,VLE, REV
	ili9325_write_reg(dw74fb,ILI9325_VER_SCROLL_CTRL, 0x0000); // set scrolling line

	/* Set partial display control */
	ili9325_write_reg(dw74fb,ILI9325_PART_IMG1_DISP_POS, 0x0000);
	ili9325_write_reg(dw74fb,ILI9325_PART_IMG1_AREA_START, 0x0000);
	ili9325_write_reg(dw74fb,ILI9325_PART_IMG1_AREA_END, 0x0000);
	ili9325_write_reg(dw74fb,ILI9325_PART_IMG2_DISP_POS, 0x0000);
	ili9325_write_reg(dw74fb,ILI9325_PART_IMG2_AREA_START, 0x0000);
	ili9325_write_reg(dw74fb,ILI9325_PART_IMG2_AREA_END, 0x0000);

	/* Set panel control */
	ili9325_write_reg(dw74fb,ILI9325_PANEL_INTF_CTRL1, 0x0010);
	ili9325_write_reg(dw74fb,ILI9325_PANEL_INTF_CTRL2, 0x0600);
//  ili9325_write_reg(dw74fb,0x0093, 0x0003); need to check Sanka
	ili9325_write_reg(dw74fb,ILI9325_PANEL_INTF_CTRL3, 0x0110);
//  ili9325_write_reg(dw74fb,0x0097, 0x0000); need to check Sanka
//  ili9325_write_reg(dw74fb,0x0098, 0x0000); need to check
	ili9325_write_reg(dw74fb,ILI9325_DISPLAY_CTRL1, 0x0133); // 262K color and display ON
  
	/* Set Index register for data transfer mode */
	ili9325_regindex(dw74fb, 0x22);
	
	//dw74fb_set_cputype(dw74fb, &ili9325);

	return 0;
}

static int __devinit dw_ili9325_probe(struct platform_device *pdev)
{
	struct ili9325_plat_data *pdata;
	struct dw74fb *dw74fb = dev_get_drvdata(&pdev->dev);
    
	pdata = kzalloc(sizeof(struct ili9325_plat_data), GFP_KERNEL);
	if (!pdata)
		return -ENOMEM;

	pdata->dw74fb = dw74fb;
	pdev->dev.platform_data = pdata;

	#ifdef CONFIG_HAS_EARLYSUSPEND
		pdata->ili9325_early_suspend.suspend = ili9325_early_suspend;
		pdata->ili9325_early_suspend.resume = ili9325_late_resume;
		pdata->ili9325_early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;
		register_early_suspend(&pdata->ili9325_early_suspend);
	#endif

	ili9325_init(dw74fb);

	return 0;
}


static int __devexit dw_ili9325_remove(struct platform_device *pdev)
{
	struct ili9325_plat_data *pdata = pdev->dev.platform_data;

	unregister_early_suspend(&pdata->ili9325_early_suspend);

	kfree(pdev->dev.platform_data);
    return 0;
}


static struct platform_driver dw_ili9325_driver = {
	.probe = dw_ili9325_probe,
	.remove = __exit_p(dw_ili9325_remove),
	.driver = {
		.name = "dw-ili9325",
		.owner = THIS_MODULE,
	},
};

static int __init dw_ili9325_init(void)
{
    return platform_driver_register(&dw_ili9325_driver);
}

static void __exit dw_ili9325_exit(void)
{
	platform_driver_unregister(&dw_ili9325_driver);
}

module_init(dw_ili9325_init);
module_exit(dw_ili9325_exit);

MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD ili9325");
