/*
 * drivers/power/dp52-power.c - DCINS power driver for DP52 chip
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/slab.h>

#include <linux/mfd/dp52/core.h>
#include <linux/mfd/dp52/power.h>

#define ONLINE_POLL_SEC			1
#define USER_SPACE_UPDATE_SEC	20
#define POLL_INTERVAL_ONLINE	((ONLINE_POLL_SEC)*HZ)
#define POLL_INTERVAL_OFFLINE	((USER_SPACE_UPDATE_SEC)*HZ)

struct dp52_power {
	struct device *dev;
	struct dp52 *dp52;
	struct dp52_power_pdata *pdata;
	int irq;
	struct power_supply psy;
	struct timer_list poll_timer;
	struct work_struct poll_work;
};

static int is_power_online(struct dp52_power *power)
{
	int ret = !!(dp52_read(power->dp52, DP52_PMU_RAWSTAT) & 0x02);

	if (power->pdata && power->pdata->filter)
		ret = power->pdata->filter(power->dp52, ret,
		                           power->pdata->filter_priv);

	return ret;
}

static int dp52_power_get_property(struct power_supply *psy,
                                   enum power_supply_property psp,
                                   union power_supply_propval *val)
{
	struct dp52_power *power = container_of(psy, struct dp52_power, psy);
	int online;

	switch (psp) {
		case POWER_SUPPLY_PROP_ONLINE:
			online = is_power_online(power);
			if (online)
				mod_timer(&power->poll_timer, jiffies + POLL_INTERVAL_ONLINE);
			val->intval = online;
			break;

		default:
			return -EINVAL;
	}

	return 0;
}

static enum power_supply_property dp52_power_props[] = {
	POWER_SUPPLY_PROP_ONLINE,
};

static char *dp52_power_supplied_to[] = {
	"main-battery"
};

static void poll_timeout(unsigned long priv)
{
	struct dp52_power *power = (struct dp52_power *)priv;

	schedule_work(&power->poll_work);
}

static void poll_work(struct work_struct *work)
{
	static int update_cnt = 0;
	int changed = 0;
	int online;
	int interval;
	struct dp52_power *power = container_of(work, struct dp52_power, poll_work);

	online = is_power_online(power);

	/* update user-space if we are not online any more, */
	/* or if USER_SPACE_UPDATE_SEC seconds expired while online */
	if (!online || update_cnt++ == ((USER_SPACE_UPDATE_SEC)/(ONLINE_POLL_SEC))) {
		changed = 1;
	}

	interval = online ? POLL_INTERVAL_ONLINE : POLL_INTERVAL_OFFLINE; 
	mod_timer(&power->poll_timer, jiffies + interval);

	if (changed) {
		power_supply_changed(&power->psy);
		update_cnt = 0;
	}
}

static irqreturn_t dp52_power_irq(int irq, void *priv)
{
	struct dp52_power *power = priv;
	int pending;

	pending = dp52_read(power->dp52, DP52_PMU_STAT);

	if (pending & (1 << 1)) {
		power_supply_changed(&power->psy);
		dp52_write(power->dp52, DP52_PMU_STAT, ~(1 << 1));
		return IRQ_HANDLED;
	}

	return IRQ_NONE;
}

static int __devinit dp52_power_probe(struct platform_device *pdev)
{
	struct dp52 *dp52 = dev_get_drvdata(pdev->dev.parent);
	struct dp52_power_pdata *pdata = pdev->dev.platform_data;
	struct dp52_power *power;
	int ret;

	power = kzalloc(sizeof(*power), GFP_KERNEL);
	if (!power)
		return -ENOMEM;

	device_init_wakeup(&pdev->dev, 1);

	platform_set_drvdata(pdev, power);

	power->dev = &pdev->dev;
	power->dp52 = dp52;
	power->pdata = pdata;
	INIT_WORK(&power->poll_work, poll_work);
	setup_timer(&power->poll_timer, poll_timeout, (unsigned long)power);
	if ((power->irq = platform_get_irq_byname(pdev, "pmu")) < 0) {
		ret = power->irq;
		dev_err(&pdev->dev, "no IRQ!\n");
		goto err_free;
	}

	power->psy.name = "ac";
	power->psy.type = POWER_SUPPLY_TYPE_MAINS;
	power->psy.properties = dp52_power_props;
	power->psy.num_properties = ARRAY_SIZE(dp52_power_props);
	power->psy.get_property = dp52_power_get_property;
	if (pdata && pdata->supplied_to) {
		power->psy.supplied_to = pdata->supplied_to;
		power->psy.num_supplicants = pdata->num_supplicants;
	} else {
		power->psy.supplied_to = dp52_power_supplied_to;
		power->psy.num_supplicants = ARRAY_SIZE(dp52_power_supplied_to);
	}

	if ((ret = power_supply_register(&pdev->dev, &power->psy))) {
		dev_err(&pdev->dev, "failed to register power supply\n");
		goto err_free;
	}

	ret = request_threaded_irq(power->irq, NULL, dp52_power_irq, IRQF_SHARED,
	                           "dp52-power", power);
	if (ret) {
		dev_err(&pdev->dev, "failed to request IRQ %d\n", power->irq);
		goto err_unregister;
	}

	/* enable DCINS interrupt */
	dp52_set_bits(dp52, DP52_PMU_INTMSK, 1 << 1);

	mod_timer(&power->poll_timer, jiffies + POLL_INTERVAL_OFFLINE);

	dev_info(&pdev->dev, "initialized\n");
	return 0;

err_unregister:
	power_supply_unregister(&power->psy);
err_free:
	kfree(power);
	return ret;
}

static int __devexit dp52_power_remove(struct platform_device *pdev)
{
	struct dp52_power *power = platform_get_drvdata(pdev);

	free_irq(power->irq, power);
	del_timer_sync(&power->poll_timer);
	flush_scheduled_work();
	power_supply_unregister(&power->psy);
	kfree(power);

	return 0;
}

static void dp52_power_shutdown(struct platform_device *pdev)
{
	struct dp52_power *power = platform_get_drvdata(pdev);

	dp52_clr_bits(power->dp52, DP52_PMU_INTMSK, 1 << 1);
}

static struct platform_driver dp52_power_driver = {
	.driver = {
		.name  = "dp52-power",
		.owner = THIS_MODULE,
	},
	.probe  = dp52_power_probe,
	.remove = __devexit_p(dp52_power_remove),
	.shutdown = dp52_power_shutdown,
};

static int __init dp52_power_init(void)
{
	return platform_driver_register(&dp52_power_driver);
}

static void __exit dp52_power_exit(void)
{
	platform_driver_unregister(&dp52_power_driver);
}

module_init(dp52_power_init);
module_exit(dp52_power_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("DSP Group Inc.");
MODULE_DESCRIPTION("DP52 DCINS power supply driver");
MODULE_ALIAS("platform:dp52-power");

