/* drivers/input/touchscreen/pixcir_i2c_ts.c
 *
 * Copyright (C) 2010 Pixcir, Inc.
 *
 * pixcir_i2c_ts.c V1.0  support multi touch
 * pixcir_i2c_ts.c V1.5  add Calibration function:
 *
 * CALIBRATION_FLAG	1
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/i2c/pixcir-ts.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/slab.h>

#include "pixcir_i2c_ts.h"

#define DRIVER_VERSION "v1.5"
#define DRIVER_AUTHOR "Bee<http://www.pixcir.com.cn>"
#define DRIVER_DESC "Pixcir I2C Touchscreen Driver with tune fuction"
#define DRIVER_LICENSE "GPL"

#define TWO_POINTS

#define SLAVE_ADDR		0x5c

#ifndef I2C_MAJOR
#define I2C_MAJOR		125
#endif

#define I2C_MINORS		256

#define CALIBRATION_FLAG	1

#define FLIP_X
#define FLIP_Y
#define NO_TOUCH_FIX

//#define DEBUG

#ifdef DEBUG
  #define PIXDEBUG(x...) printk(x)
#else
  #define PIXDEBUG(x...)
#endif

static unsigned char status_reg = 0;

struct i2c_dev
{
	struct list_head list;
	struct i2c_adapter *adap;
	struct device *dev;
};

static struct i2c_driver pixcir_i2c_ts_driver;
static struct class *i2c_dev_class;
static LIST_HEAD(i2c_dev_list);
static DEFINE_SPINLOCK(i2c_dev_list_lock);

static struct workqueue_struct *pixcir_wq;

struct pixcir_i2c_ts_data
{
	struct i2c_client *client;
	struct input_dev *input;
	struct delayed_work work;
#ifdef NO_TOUCH_FIX
	struct delayed_work verify;
#endif
	struct pixcir168_platform_data *platform_data;
	int irq;

	unsigned char touching;
};

inline int convert_x(struct pixcir168_platform_data *pd, int x)
{
	if (!pd)
		return x;

	/* interpolate x from 0 .. TOUCHSCREEN_MAXX-1 to 0 .. pd->xres_full-1*/
	x = x * pd->xres_full / TOUCHSCREEN_MAXX;

	/* handle xres inside xres_full */
	return x - ((pd->xres_full - pd->xres) / 2);
}

inline int getX(struct pixcir168_platform_data *pd, int x)
{
#ifdef FLIP_X
	x = TOUCHSCREEN_MAXX - x;
#endif

	return convert_x(pd, x);
}

inline int convert_y(struct pixcir168_platform_data *pd, int y)
{
	if (!pd)
		return y;

	/* interpolate y from 0 .. TOUCHSCREEN_MAXY-1 to 0 .. pd->yres_full-1*/
	y = y * pd->yres_full / TOUCHSCREEN_MAXY;

	/* handle yres inside yres_full */
	return y - ((pd->yres_full - pd->yres) / 2);
}

inline int getY(struct pixcir168_platform_data *pd, int y)
{
#ifdef FLIP_Y
	y = TOUCHSCREEN_MAXY - y;
#endif

	return convert_y(pd, y);
}

static void
return_i2c_dev(struct i2c_dev *i2c_dev)
{
	spin_lock(&i2c_dev_list_lock);
	list_del(&i2c_dev->list);
	spin_unlock(&i2c_dev_list_lock);
	kfree(i2c_dev);
}

static struct i2c_dev *
i2c_dev_get_by_minor(unsigned index)
{
	struct i2c_dev *i2c_dev;
	i2c_dev = NULL;

	spin_lock(&i2c_dev_list_lock);
	list_for_each_entry(i2c_dev, &i2c_dev_list, list) {
		if (i2c_dev->adap->nr == index)
		goto found;
	}
	i2c_dev = NULL;
found:
	spin_unlock(&i2c_dev_list_lock);
	return i2c_dev;
}

static struct i2c_dev *
get_free_i2c_dev(struct i2c_adapter *adap)
{
	struct i2c_dev *i2c_dev;

	if (adap->nr >= I2C_MINORS) {
		printk(KERN_ERR "i2c-dev: Out of device minors (%d)\n",
				adap->nr);
		return ERR_PTR(-ENODEV);
	}

	i2c_dev = kzalloc(sizeof(*i2c_dev), GFP_KERNEL);
	if (!i2c_dev)
		return ERR_PTR(-ENOMEM);
	i2c_dev->adap = adap;

	spin_lock(&i2c_dev_list_lock);
	list_add_tail(&i2c_dev->list, &i2c_dev_list);
	spin_unlock(&i2c_dev_list_lock);
	return i2c_dev;
}

#ifdef NO_TOUCH_FIX
static void no_touch_verify(struct work_struct *work)
{
	int ret;
	unsigned char touching;
	unsigned char readBuf[2], writeBuf[1];
	struct pixcir_i2c_ts_data *tsdata = container_of(work,
			struct pixcir_i2c_ts_data,
			verify.work);

	memset(readBuf, 0, sizeof(readBuf));
	memset(writeBuf, 0, sizeof(writeBuf));

	writeBuf[0] = 0;
	ret = i2c_master_send(tsdata->client, writeBuf, 1);
	if (ret != 1) {
		dev_err(&tsdata->client->dev, "Unable to write to i2c touchscreen!\n");
		return;
	}

	ret = i2c_master_recv(tsdata->client, readBuf, sizeof(readBuf));

	if (ret != sizeof(readBuf)) {
		dev_err(&tsdata->client->dev, "Unable to read i2c page!\n");
		return;
	}

	touching = readBuf[0];

	if (tsdata->touching && !touching) {
		PIXDEBUG("missed finger lift. reporting now.\n");

		// last report was that we are still touching but we are not
		input_report_key(tsdata->input, BTN_TOUCH, 0);
		input_report_abs(tsdata->input, ABS_PRESSURE, 0);

		input_sync(tsdata->input);
	}

}
#endif

static void
pixcir_ts_poscheck(struct work_struct *work)
{
	struct pixcir_i2c_ts_data *tsdata = container_of(work,
			struct pixcir_i2c_ts_data,
			work.work);
	unsigned char touching = 0;
	unsigned char oldtouching = 0;
	int posx1, posy1, posx2, posy2;
	unsigned char Rdbuf[10],Wrbuf[1];
#ifdef R8C_AUO_I2C
	unsigned char auotpnum[1];
#endif
	int ret;
	int z = 50;
	int w = 15;

	memset(Wrbuf, 0, sizeof(Wrbuf));
	memset(Rdbuf, 0, sizeof(Rdbuf));

	Wrbuf[0] = 0;

	ret = i2c_master_send(tsdata->client, Wrbuf, 1);

	if (ret != 1) {
		dev_err(&tsdata->client->dev, "Unable to write to i2c touchscreen!\n");
		goto out;
	}

#ifdef R8C_AUO_I2C
	ret = i2c_master_recv(tsdata->client, Rdbuf, 8);

	if (ret != 8) {
		dev_err(&tsdata->client->dev, "Unable to read i2c page!\n");
		goto out;
	}

	posx1 = ((Rdbuf[1] << 8) | Rdbuf[0]);
	posy1 = ((Rdbuf[3] << 8) | Rdbuf[2]);
	posx2 = ((Rdbuf[5] << 8) | Rdbuf[4]);
	posy2 = ((Rdbuf[7] << 8) | Rdbuf[6]);
#else
	ret = i2c_master_recv(tsdata->client, Rdbuf, sizeof(Rdbuf));

	if (ret != sizeof(Rdbuf)) {
		dev_err(&tsdata->client->dev, "Unable to read i2c page!\n");
		goto out;
	}

	posx1 = ((Rdbuf[3] << 8) | Rdbuf[2]);
	posy1 = ((Rdbuf[5] << 8) | Rdbuf[4]);
	posx2 = ((Rdbuf[7] << 8) | Rdbuf[6]);
	posy2 = ((Rdbuf[9] << 8) | Rdbuf[8]);
#endif

#ifdef R8C_AUO_I2C
	if ((posx1 != 0) || (posy1 != 0) || (posx2 != 0) || (posy2 != 0)) {
		Wrbuf[0] = 113;
		ret = i2c_master_send(tsdata->client, Wrbuf, 1);
		ret = i2c_master_recv(tsdata->client, auotpnum, 1);
		touching = auotpnum[0]>>5;
		oldtouching = 0;
	}
#else
	tsdata->touching = touching = Rdbuf[0];
	oldtouching = Rdbuf[1];
#endif

	PIXDEBUG("touching:%-3d,oldtouching:%-3d,x1:%-6d,y1:%-6d,x2:%-6d,y2:%-6d\n",touching, oldtouching, posx1, posy1, posx2, posy2);

	if (touching) {
		input_report_abs(tsdata->input, ABS_X, getX(tsdata->platform_data, posx1));
		input_report_abs(tsdata->input, ABS_Y, getY(tsdata->platform_data, posy1));
		input_report_key(tsdata->input, BTN_TOUCH, 1);
		input_report_abs(tsdata->input, ABS_PRESSURE, 1);
	} else {
		input_report_key(tsdata->input, BTN_TOUCH, 0);
		input_report_abs(tsdata->input, ABS_PRESSURE, 0);
	}

	if (!(touching)) {
		z = 0;
		w = 0;
	}
#ifdef TWO_POINTS
	input_report_abs(tsdata->input, ABS_MT_TOUCH_MAJOR, z);
	input_report_abs(tsdata->input, ABS_MT_WIDTH_MAJOR, w);
	input_report_abs(tsdata->input, ABS_MT_POSITION_X, getX(tsdata->platform_data, posx1));
	input_report_abs(tsdata->input, ABS_MT_POSITION_Y, getY(tsdata->platform_data, posy1));
	input_mt_sync(tsdata->input);

	if (touching == 2) {
		input_report_abs(tsdata->input, ABS_MT_TOUCH_MAJOR, z);
		input_report_abs(tsdata->input, ABS_MT_WIDTH_MAJOR, w);
		input_report_abs(tsdata->input, ABS_MT_POSITION_X, getX(tsdata->platform_data, posx2));
		input_report_abs(tsdata->input, ABS_MT_POSITION_Y, getY(tsdata->platform_data, posy2));
		input_mt_sync(tsdata->input);
	}
#endif
	input_sync(tsdata->input);

out:
	enable_irq(tsdata->irq);

#ifdef NO_TOUCH_FIX
	cancel_delayed_work_sync(&tsdata->verify);
	queue_delayed_work(pixcir_wq, &tsdata->verify, HZ >> 5);
#endif
}

static irqreturn_t
pixcir_ts_isr(int irq, void *dev_id)
{
	struct pixcir_i2c_ts_data *tsdata = dev_id;

	disable_irq_nosync(irq);
	queue_work(pixcir_wq, &tsdata->work.work);

	return IRQ_HANDLED;
}

static int
pixcir_ts_open(struct input_dev *dev)
{
	return 0;
}

static void
pixcir_ts_close(struct input_dev *dev)
{
}

static int
pixcir_i2c_ts_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct pixcir_i2c_ts_data *tsdata;
	struct input_dev *input;
	struct device *dev;
	struct i2c_dev *i2c_dev;
	int xres;
	int yres;
	int error;

	tsdata = kzalloc(sizeof(*tsdata), GFP_KERNEL);
	if (!tsdata) {
		dev_err(&client->dev, "failed to allocate driver data!\n");
		error = -ENOMEM;
		dev_set_drvdata(&client->dev, NULL);
		return error;
	}

	dev_set_drvdata(&client->dev, tsdata);

	input = input_allocate_device();
	if (!input) {
		dev_err(&client->dev, "failed to allocate input device!\n");
		error = -ENOMEM;
		input_free_device(input);
		kfree(tsdata);
	}

	tsdata->platform_data = (struct pixcir168_platform_data*)client->dev.platform_data;
	if (tsdata->platform_data) {
		if (!tsdata->platform_data->xres)
			tsdata->platform_data->xres = TOUCHSCREEN_MAXX;
		if (!tsdata->platform_data->yres)
			tsdata->platform_data->yres = TOUCHSCREEN_MAXY;
		if (!tsdata->platform_data->xres_full)
			tsdata->platform_data->xres_full = tsdata->platform_data->xres;
		if (!tsdata->platform_data->yres_full)
			tsdata->platform_data->yres_full = tsdata->platform_data->yres;

		xres = tsdata->platform_data->xres;
		yres = tsdata->platform_data->yres;
	}
	else {
		xres = TOUCHSCREEN_MAXX;
		yres = TOUCHSCREEN_MAXY;
	}

	set_bit(EV_SYN, input->evbit);
	set_bit(EV_KEY, input->evbit);
	set_bit(EV_ABS, input->evbit);
	set_bit(BTN_TOUCH, input->keybit);
	input_set_abs_params(input, ABS_X, 0, xres - 1, 0, 0);
	input_set_abs_params(input, ABS_Y, 0, yres - 1, 0, 0);
#ifdef TWO_POINTS
	input_set_abs_params(input, ABS_MT_POSITION_X, 0, xres - 1, 0, 0);
	input_set_abs_params(input, ABS_MT_POSITION_Y, 0, yres - 1, 0, 0);
	input_set_abs_params(input, ABS_MT_TOUCH_MAJOR, 0, 255, 0, 0);
	input_set_abs_params(input, ABS_MT_WIDTH_MAJOR, 0, 25, 0, 0);
#endif

	input->name = client->name;
	input->id.bustype = BUS_I2C;
	input->dev.parent = &client->dev;
	input->open = pixcir_ts_open;
	input->close = pixcir_ts_close;

	input_set_drvdata(input, tsdata);

	tsdata->client = client;
	tsdata->input = input;

	INIT_WORK(&tsdata->work.work, pixcir_ts_poscheck);
#ifdef NO_TOUCH_FIX
	INIT_DELAYED_WORK(&(tsdata->verify), no_touch_verify);
#endif

	tsdata->irq = client->irq;

	if (input_register_device(input)) {
		input_free_device(input);
		kfree(tsdata);
	}

	if (request_irq(tsdata->irq, pixcir_ts_isr, IRQF_TRIGGER_LOW,client->name, tsdata)) {
		dev_err(&client->dev, "Unable to request touchscreen IRQ.\n");
		input_unregister_device(input);
		input = NULL;
	}

	device_init_wakeup(&client->dev, 1);

	i2c_dev = get_free_i2c_dev(client->adapter);
	if (IS_ERR(i2c_dev)) {
		error = PTR_ERR(i2c_dev);
		return error;
	}

	dev = device_create(i2c_dev_class, &client->adapter->dev,
	                    MKDEV(I2C_MAJOR, client->adapter->nr), NULL,
			    "pixcir_i2c_ts%d", client->adapter->nr);
	if (IS_ERR(dev)) {
		error = PTR_ERR(dev);
		return error;
	}

	return 0;
}

static int
pixcir_i2c_ts_remove(struct i2c_client *client)
{
	int error;
	struct i2c_dev *i2c_dev;
	struct pixcir_i2c_ts_data *tsdata = dev_get_drvdata(&client->dev);

	free_irq(tsdata->irq, tsdata);
	i2c_dev = get_free_i2c_dev(client->adapter);
	if (IS_ERR(i2c_dev)) {
		error = PTR_ERR(i2c_dev);
		return error;
	}
	return_i2c_dev(i2c_dev);
	device_destroy(i2c_dev_class, MKDEV(I2C_MAJOR, client->adapter->nr));
	input_unregister_device(tsdata->input);
	kfree(tsdata);
	dev_set_drvdata(&client->dev, NULL);

	return 0;
}

static int
pixcir_i2c_ts_suspend(struct device *dev)
{
	struct pixcir_i2c_ts_data *tsdata = dev_get_drvdata(dev);

	if (device_may_wakeup(dev))
		enable_irq_wake(tsdata->irq);

	return 0;
}

static int
pixcir_i2c_ts_resume(struct device *dev)
{
	struct pixcir_i2c_ts_data *tsdata = dev_get_drvdata(dev);

	if (device_may_wakeup(dev))
		disable_irq_wake(tsdata->irq);

	return 0;
}

static SIMPLE_DEV_PM_OPS(pixcir_i2c_ts_dev_pm_ops,
			 pixcir_i2c_ts_suspend, pixcir_i2c_ts_resume);

static int
pixcir_open(struct inode *inode, struct file *file)
{
	int subminor;
	struct i2c_client *client;
	struct i2c_adapter *adapter;
	struct i2c_dev *i2c_dev;
	int ret = 0;

	subminor = iminor(inode);
	i2c_dev = i2c_dev_get_by_minor(subminor);
	if (!i2c_dev) {
		printk("error i2c_dev\n");
		return -ENODEV;
	}

	adapter = i2c_get_adapter(i2c_dev->adap->nr);
	if (!adapter)
		return -ENODEV;

	client = kzalloc(sizeof(*client), GFP_KERNEL);
	if (!client) {
		i2c_put_adapter(adapter);
		ret = -ENOMEM;
	}
	snprintf(client->name, I2C_NAME_SIZE, "pixcir_i2c_ts%d", adapter->nr);
	client->driver = &pixcir_i2c_ts_driver;
	client->adapter = adapter;
	//if(i2cdev_check_addr(client->adapter,0x5c))
	//	return -EBUSY;
	file->private_data = client;

	return 0;
}

static long
pixcir_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct i2c_client *client = (struct i2c_client *)file->private_data;

	switch (cmd)
	{
	case CALIBRATION_FLAG:
		client->addr = SLAVE_ADDR;
		status_reg = 0;
		status_reg = CALIBRATION_FLAG;
		break;

	default:
		break;//return -ENOTTY;
	}
	return 0;
}

static ssize_t
pixcir_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	struct i2c_client *client;
	char *tmp;
	static int ret = 0;

	client = file->private_data;

	switch(status_reg) {
	case CALIBRATION_FLAG:
		tmp = kmalloc(count,GFP_KERNEL);
		if (tmp == NULL)
			return -ENOMEM;
		if (copy_from_user(tmp, buf, count)) {
			printk("CALIBRATION_FLAG copy_from_user error\n");
			kfree(tmp);
			return -EFAULT;
		}
		ret = i2c_master_send(client, tmp, count);

		mdelay(100);
		if (ret != count)
			printk("CALIBRATION_FLAG,Unable to write to i2c page for calibratoion!\n");

		kfree(tmp);

		status_reg = 0;
		break;


		default:
		break;
	}
	return ret;
}

static int
pixcir_release(struct inode *inode, struct file *file)
{
	struct i2c_client *client = file->private_data;

	i2c_put_adapter(client->adapter);
	kfree(client);
	file->private_data = NULL;

	return 0;
}

static const struct file_operations pixcir_i2c_ts_fops =
{	.owner = THIS_MODULE,
	.write = pixcir_write,
	.open = pixcir_open,
	.unlocked_ioctl = pixcir_ioctl,
	.release = pixcir_release,
};

static const struct i2c_device_id pixcir_i2c_ts_id[] =
{
	{ "pixcir168" },
	{ }
};
MODULE_DEVICE_TABLE(i2c, pixcir_i2c_ts_id);

static struct i2c_driver pixcir_i2c_ts_driver =
{	.driver = {
		.owner = THIS_MODULE,
		.name = "pixcir_i2c_ts_driver_v1.5",
		.pm = &pixcir_i2c_ts_dev_pm_ops,
	},
	.probe = pixcir_i2c_ts_probe,
	.remove = pixcir_i2c_ts_remove,
	.id_table = pixcir_i2c_ts_id,
};

static int __init
pixcir_i2c_ts_init(void)
{
	int ret;
	pixcir_wq = create_singlethread_workqueue("pixcir_wq");
	if(!pixcir_wq)
		return -ENOMEM;

	ret = register_chrdev(I2C_MAJOR, "pixcir_i2c_ts", &pixcir_i2c_ts_fops);
	if(ret) {
		printk(KERN_ERR "%s:register chrdev failed\n", __FILE__);
		return ret;
	}

	i2c_dev_class = class_create(THIS_MODULE, "pixcir_i2c_dev");
	if (IS_ERR(i2c_dev_class)) {
		ret = PTR_ERR(i2c_dev_class);
		class_destroy(i2c_dev_class);
	}

	return i2c_add_driver(&pixcir_i2c_ts_driver);
}

static void __exit
pixcir_i2c_ts_exit(void)
{
	i2c_del_driver(&pixcir_i2c_ts_driver);
	class_destroy(i2c_dev_class);
	unregister_chrdev(I2C_MAJOR, "pixcir_i2c_ts");
	if (pixcir_wq)
		destroy_workqueue(pixcir_wq);
}

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE(DRIVER_LICENSE);

module_init(pixcir_i2c_ts_init);
module_exit(pixcir_i2c_ts_exit);
