# GNU Enterprise Forms - gtk2 UI Driver - MenuItem widget
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: menuitem.py 9956 2009-10-11 18:54:57Z reinhard $

import gtk

from gnue.forms.uidrivers.gtk2.widgets._base import UIHelper


# =============================================================================
# Wrap an UI layer around a Menu widget
# =============================================================================

class UIMenuItem(UIHelper):
    """
    Implements a menu item object.
    """

    # -------------------------------------------------------------------------
    # Create a menu item widget
    # -------------------------------------------------------------------------

    def _create_widget_(self, event, spacer):
        """
        Creates a new MenuItem widget.
        """

        # These are the relevant parameters
        icon_file = self._gfObject._get_icon_file(size="16x16", format="png")
        label = self._gfObject.label
        description = self._gfObject.description
        hotkey = self._gfObject.hotkey
        check = (self._gfObject.action_off is not None)

        if label is not None:
            if check:
                widget = gtk.CheckMenuItem(label.replace("&", "_"))
            else:
                widget = gtk.ImageMenuItem(label.replace("&", "_"))

                if icon_file:
                    image = gtk.Image()
                    image.set_from_file(icon_file)
                    widget.set_image(image)

            if hotkey is not None:
                parts = hotkey.split("+")
                if len(parts[-1]) == 1:
                    base = ord(parts[-1])
                else:
                    base = self.__keymap[parts[-1]]
                mod = 0
                if "SHIFT" in parts: mod = mod | gtk.gdk.SHIFT_MASK
                if "CTRL"  in parts: mod = mod | gtk.gdk.CONTROL_MASK
                if "ALT"   in parts: mod = mod | gtk.gdk.MOD1_MASK
                widget.add_accelerator('activate', self._uiForm.accelGroup,
                            base, mod, gtk.ACCEL_VISIBLE)
    
            event.container.append(widget)

            self.__description = description
            self.__container = event.container

            self.__h_activate = widget.connect('activate', self.__on_activate)
            widget.connect('select', self.__on_select)
            widget.connect('deselect', self.__on_deselect)
        else:
            widget = None
            event.container.add(gtk.SeparatorMenuItem())

        self.__widget = widget

        return widget


    # -------------------------------------------------------------------------
    # Events
    # -------------------------------------------------------------------------

    def __on_activate(self, event):
        self._gfObject._event_fire()

    # -------------------------------------------------------------------------

    def __on_select(self, menu_item):
        self._uiForm._show_tip(self.__description)

    # -------------------------------------------------------------------------

    def __on_deselect(self, menu_item):
        self._uiForm._show_tip("")

    # -------------------------------------------------------------------------
    # Check/uncheck menu item
    # -------------------------------------------------------------------------

    def _ui_switch_on_(self):
        if self.__widget is not None:
            self.__widget.handler_block(self.__h_activate)
            self.__widget.set_active(True)
            self.__widget.handler_unblock(self.__h_activate)

    # -------------------------------------------------------------------------

    def _ui_switch_off_(self):
        if self.__widget is not None:
            self.__widget.handler_block(self.__h_activate)
            self.__widget.set_active(False)
            self.__widget.handler_unblock(self.__h_activate)


    # -------------------------------------------------------------------------
    # Enable/disable menu item
    # -------------------------------------------------------------------------

    def _ui_enable_(self):
        if self.__widget is not None:
            self.__widget.set_sensitive(True)

    # -------------------------------------------------------------------------

    def _ui_disable_(self):
        if self.__widget is not None:
            self.__widget.set_sensitive(False)


    # -------------------------------------------------------------------------
    # Keymap translation
    # -------------------------------------------------------------------------

    __keymap = {
        "F1"        : gtk.keysyms.F1,
        "F2"        : gtk.keysyms.F2,
        "F3"        : gtk.keysyms.F3,
        "F4"        : gtk.keysyms.F4,
        "F5"        : gtk.keysyms.F5,
        "F6"        : gtk.keysyms.F6,
        "F7"        : gtk.keysyms.F7,
        "F8"        : gtk.keysyms.F8,
        "F9"        : gtk.keysyms.F9,
        "F10"       : gtk.keysyms.F10,
        "F11"       : gtk.keysyms.F11,
        "F12"       : gtk.keysyms.F12,
        "INS"       : gtk.keysyms.Insert,
        "DEL"       : gtk.keysyms.Delete,
        "HOME"      : gtk.keysyms.Home,
        "END"       : gtk.keysyms.End,
        "PGUP"      : gtk.keysyms.Prior,
        "PGDN"      : gtk.keysyms.Next,
        "UP"        : gtk.keysyms.Up,
        "DOWN"      : gtk.keysyms.Down,
        "LEFT"      : gtk.keysyms.Left,
        "RIGHT"     : gtk.keysyms.Right,
        "TAB"       : gtk.keysyms.Tab,
        "ENTER"     : gtk.keysyms.Return,
        "BACK"      : gtk.keysyms.BackSpace}

# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIMenuItem,
  'provides' : 'GFMenuItem',
  'container': False
}
