/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    RandomCommittee.java
 *    Copyright (C) 2003-2012 University of Waikato, Hamilton, New Zealand
 *
 */

package weka.classifiers.meta;

import java.util.HashSet;
import java.util.Random;
import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import weka.classifiers.AbstractClassifier;
import weka.classifiers.RandomizableParallelIteratedSingleClassifierEnhancer;
import weka.core.*;
import weka.filters.Filter;

/**
 <!-- globalinfo-start -->
 * Class for building an ensemble of randomizable base classifiers. Each base classifiers is built using a different random number seed (but based one the same data). The final prediction is a straight average of the predictions generated by the individual base classifiers.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre> -S &lt;num&gt;
 *  Random number seed.
 *  (default 1)</pre>
 * 
 * <pre> -I &lt;num&gt;
 *  Number of iterations.
 *  (default 10)</pre>
 * 
 * <pre> -D
 *  If set, classifier is run in debug mode and
 *  may output additional info to the console</pre>
 * 
 * <pre> -W
 *  Full name of base classifier.
 *  (default: weka.classifiers.trees.RandomTree)</pre>
 * 
 * <pre> 
 * Options specific to classifier weka.classifiers.trees.RandomTree:
 * </pre>
 * 
 * <pre> -K &lt;number of attributes&gt;
 *  Number of attributes to randomly investigate
 *  (&lt;1 = int(log(#attributes)+1)).</pre>
 * 
 * <pre> -M &lt;minimum number of instances&gt;
 *  Set minimum number of instances per leaf.</pre>
 * 
 * <pre> -S &lt;num&gt;
 *  Seed for random number generator.
 *  (default 1)</pre>
 * 
 * <pre> -depth &lt;num&gt;
 *  The maximum depth of the tree, 0 for unlimited.
 *  (default 0)</pre>
 * 
 * <pre> -D
 *  If set, classifier is run in debug mode and
 *  may output additional info to the console</pre>
 * 
 <!-- options-end -->
 *
 * Options after -- are passed to the designated classifier.<p>
 *
 * @author Eibe Frank (eibe@cs.waikato.ac.nz)
 * @version $Revision: 15801 $
 */
public class RandomCommittee 
  extends RandomizableParallelIteratedSingleClassifierEnhancer
  implements WeightedInstancesHandler, PartitionGenerator {
    
  /** for serialization */
  static final long serialVersionUID = -9204394360557300093L;
  
  /** training data */
  protected Instances m_data;
  
  /**
   * Constructor.
   */
  public RandomCommittee() {
    
    m_Classifier = new weka.classifiers.trees.RandomTree();
  }

  /**
   * String describing default classifier.
   * 
   * @return the default classifier classname
   */
  protected String defaultClassifierString() {
    
    return "weka.classifiers.trees.RandomTree";
  }

  /**
   * Returns a string describing classifier
   * @return a description suitable for
   * displaying in the explorer/experimenter gui
   */
  public String globalInfo() {
 
    return "Class for building an ensemble of randomizable base classifiers. Each "
      + "base classifiers is built using a different random number seed (but based "
      + "one the same data). The final prediction is a straight average of the "
      + "predictions generated by the individual base classifiers.";
  }

  /**
   * Builds the committee of randomizable classifiers.
   *
   * @param data the training data to be used for generating the
   * bagged classifier.
   * @exception Exception if the classifier could not be built successfully
   */
  public void buildClassifier(Instances data) throws Exception {

    // can classifier handle the data?
    getCapabilities().testWithFail(data);

    // get fresh instances
    m_data = new Instances(data);
    super.buildClassifier(m_data);
    
    if (!(m_Classifier instanceof Randomizable)) {
      throw new IllegalArgumentException("Base learner must implement Randomizable!");
    }

    m_Classifiers = AbstractClassifier.makeCopies(m_Classifier, m_NumIterations);

    Random random = m_data.getRandomNumberGenerator(m_Seed);

    // Resample data based on weights if base learner can't handle weights
    if (!(m_Classifier instanceof WeightedInstancesHandler) && !m_data.allInstanceWeightsIdentical()) {
      m_data = m_data.resampleWithWeights(random);
    }

    for (int j = 0; j < m_Classifiers.length; j++) {

      // Set the random number seed for the current classifier.
      ((Randomizable) m_Classifiers[j]).setSeed(random.nextInt());
      
      // Build the classifier.
//      m_Classifiers[j].buildClassifier(m_data);
    }
    
    buildClassifiers();
    
    // save memory
    m_data = null;
  }
  
  /**
   * Returns a training set for a particular iteration.
   * 
   * @param iteration the number of the iteration for the requested training set.
   * @return the training set for the supplied iteration number
   * @throws Exception if something goes wrong when generating a training set.
   */
  protected synchronized Instances getTrainingSet(int iteration) throws Exception {
    
    // we don't manipulate the training data in any way.
    return m_data;
  }

  /**
   * Calculates the class membership probabilities for the given test
   * instance.
   *
   * @param instance the instance to be classified
   * @return preedicted class probability distribution
   * @exception Exception if distribution can't be computed successfully 
   */
  public double[] distributionForInstance(Instance instance) throws Exception {

    double[] sums = new double[instance.numClasses()], newProbs;

    double numPreds = 0;
    for (int i = 0; i < m_NumIterations; i++) {
      if (instance.classAttribute().isNumeric() == true) {
        double pred = m_Classifiers[i].classifyInstance(instance);
        if (!Utils.isMissingValue(pred)) {
          sums[0] += pred;
          numPreds++;
        }
      } else {
        newProbs = m_Classifiers[i].distributionForInstance(instance);
        for (int j = 0; j < newProbs.length; j++)
          sums[j] += newProbs[j];
      }
    }
    if (instance.classAttribute().isNumeric() == true) {
      if (numPreds == 0) {
        sums[0] = Utils.missingValue();
      } else {
        sums[0] /= numPreds;
      }
      return sums;
    } else if (Utils.eq(Utils.sum(sums), 0)) {
      return sums;
    } else {
      Utils.normalize(sums);
      return sums;
    }
  }

  /**
   * Tool tip text for this property
   *
   * @return the tool tip for this property
   */
  public String batchSizeTipText() {
    return "Batch size to use if base learner is a BatchPredictor";
  }

  /**
   * Set the batch size to use. Gets passed through to the base learner if it
   * implements BatchPredictor. Otherwise it is just ignored.
   *
   * @param size the batch size to use
   */
  public void setBatchSize(String size) {

    if (getClassifier() instanceof BatchPredictor) {
      ((BatchPredictor) getClassifier()).setBatchSize(size);
    } else {
      super.setBatchSize(size);
    }
  }

  /**
   * Gets the preferred batch size from the base learner if it implements
   * BatchPredictor. Returns 1 as the preferred batch size otherwise.
   *
   * @return the batch size to use
   */
  public String getBatchSize() {

    if (getClassifier() instanceof BatchPredictor) {
      return ((BatchPredictor) getClassifier()).getBatchSize();
    } else {
      return super.getBatchSize();
    }
  }

  /**
   * Batch scoring method. Calls the appropriate method for the base learner if
   * it implements BatchPredictor. Otherwise it simply calls the
   * distributionForInstance() method repeatedly.
   *
   * @param insts the instances to get predictions for
   * @return an array of probability distributions, one for each instance
   * @throws Exception if a problem occurs
   */
  public double[][] distributionsForInstances(Instances insts) throws Exception {

    if (getClassifier() instanceof BatchPredictor) {

      ExecutorService pool = Executors.newFixedThreadPool(m_numExecutionSlots);

      // Set up result set, and chunk size
      final int chunksize = m_Classifiers.length / m_numExecutionSlots;
      Set<Future<double[][]>> results = new HashSet<Future<double[][]>>();

      // For each thread
      for (int j = 0; j < m_numExecutionSlots; j++) {

        // Determine batch to be processed
        final int lo = j * chunksize;
        final int hi = (j < m_numExecutionSlots - 1) ? (lo + chunksize) : m_Classifiers.length;

        // Create and submit new job for each batch of instances
        Future<double[][]> futureT = pool.submit(new Callable<double[][]>() {
          @Override
          public double[][] call() throws Exception {
            if (insts.classAttribute().isNumeric()) {
              double[][] ensemblePreds = new double[insts.numInstances()][2];
              for (int i = lo; i < hi; i++) {
                double[][] preds = ((BatchPredictor) m_Classifiers[i]).distributionsForInstances(insts);
                for (int j = 0; j < preds.length; j++) {
                  if (!Utils.isMissingValue(preds[j][0])) {
                    ensemblePreds[j][0] += preds[j][0];
                    ensemblePreds[j][1]++;
                  }
                }
              }
              return ensemblePreds;
            } else {
              double[][] ensemblePreds = new double[insts.numInstances()][insts.numClasses()];
              for (int i = lo; i < hi; i++) {
                double[][] preds = ((BatchPredictor) m_Classifiers[i]).distributionsForInstances(insts);
                for (int j = 0; j < preds.length; j++) {
                  for (int k = 0; k < preds[j].length; k++) {
                    ensemblePreds[j][k] += preds[j][k];
                  }
                }
              }
              return ensemblePreds;
            }
          }
        });
        results.add(futureT);
      }

      // Form ensemble prediction
      double[][] ensemblePreds =
              new double[insts.numInstances()][insts.classAttribute().isNumeric() ? 2 : insts.numClasses()];
      try {
        for (Future<double[][]> futureT : results) {
          double[][] preds = futureT.get();
          for (int j = 0; j < preds.length; j++) {
            for (int k = 0; k < preds[j].length; k++) {
              ensemblePreds[j][k] += preds[j][k];
            }
          }
        }
      } catch (Exception e) {
        System.out.println("RandomCommittee: predictions could not be generated by thread.");
        e.printStackTrace();
      }
      pool.shutdown();

      // Normalise ensemble predictions
      if (insts.classAttribute().isNumeric() == true) {
        double[][] finalPreds = new double[ensemblePreds.length][1];
        for (int j = 0; j < ensemblePreds.length; j++) {
          if (ensemblePreds[j][1] == 0) {
            finalPreds[j][0] = Utils.missingValue();
          } else {
            finalPreds[j][0] = ensemblePreds[j][0] / ensemblePreds[j][1];
          }
        }
        return finalPreds;
      } else {
        for (int j = 0; j < ensemblePreds.length; j++) {
          double sum = Utils.sum(ensemblePreds[j]);
          if (!Utils.eq((sum), 0)) {
            Utils.normalize(ensemblePreds[j], sum);
          }
        }
        return ensemblePreds;
      }
    } else {

      /** Multi-threading in this branch causes issues

      // Set up result set, and chunk size
      final int chunksize = insts.numInstances() / m_numExecutionSlots;
      Set<Future<Void>> results = new HashSet<Future<Void>>();

      final double[][] ensemblePreds = new double[insts.numInstances()][insts.numClasses()];

      // For each thread
      for (int j = 0; j < m_numExecutionSlots; j++) {

        // Determine batch to be processed
        final int lo = j * chunksize;
        final int hi = (j < m_numExecutionSlots - 1) ? (lo + chunksize) : insts.numInstances();

        // Create and submit new job for each batch of instances
        Future<Void> futureT = pool.submit(new Callable<Void>() {
          @Override
          public Void call() throws Exception {
            for (int i = lo; i < hi; i++) {
              System.arraycopy(distributionForInstance((Instance)insts.instance(i).copy()), 0,
                      ensemblePreds[i], 0, insts.numClasses());
            }
            return null;
          }
        });
        results.add(futureT);
      }

      // Incorporate predictions
      try {
        for (Future<Void> futureT : results) {
          futureT.get();
        }
      } catch (Exception e) {
        System.out.println("RandomCommittee: predictions could not be generated by thread.");
        e.printStackTrace();
      }
      pool.shutdown();

      return ensemblePreds;*/
      double[][] result = new double[insts.numInstances()][insts.numClasses()];
      for (int i = 0; i < insts.numInstances(); i++) {
        result[i] = distributionForInstance(insts.instance(i));
      }
      return result;
    }
  }

  /**
   * Returns true if the base classifier implements BatchPredictor and is able
   * to generate batch predictions efficiently
   *
   * @return true if the base classifier can generate batch predictions efficiently
   */
  public boolean implementsMoreEfficientBatchPrediction() {
    if (!(getClassifier() instanceof BatchPredictor)) {
      return super.implementsMoreEfficientBatchPrediction();
    }
    return ((BatchPredictor) getClassifier()).implementsMoreEfficientBatchPrediction();
  }

  /**
   * Returns description of the committee.
   *
   * @return description of the committee as a string
   */
  public String toString() {
    
    if (m_Classifiers == null) {
      return "RandomCommittee: No model built yet.";
    }
    StringBuffer text = new StringBuffer();
    text.append("All the base classifiers: \n\n");
    for (int i = 0; i < m_Classifiers.length; i++)
      text.append(m_Classifiers[i].toString() + "\n\n");

    return text.toString();
  }
    
  /**
   * Builds the classifier to generate a partition.
   */
  public void generatePartition(Instances data) throws Exception {
    
    if (m_Classifier instanceof PartitionGenerator)
      buildClassifier(data);
    else throw new Exception("Classifier: " + getClassifierSpec()
			     + " cannot generate a partition");
  }
  
  /**
   * Computes an array that indicates leaf membership
   */
  public double[] getMembershipValues(Instance inst) throws Exception {
    
    if (m_Classifier instanceof PartitionGenerator) {
      ArrayList<double[]> al = new ArrayList<double[]>();
      int size = 0;
      for (int i = 0; i < m_Classifiers.length; i++) {
        double[] r = ((PartitionGenerator)m_Classifiers[i]).
          getMembershipValues(inst);
        size += r.length;
        al.add(r);
      }
      double[] values = new double[size];
      int pos = 0;
      for (double[] v: al) {
        System.arraycopy(v, 0, values, pos, v.length);
        pos += v.length;
      }
      return values;
    } else throw new Exception("Classifier: " + getClassifierSpec()
                               + " cannot generate a partition");
  }
  
  /**
   * Returns the number of elements in the partition.
   */
  public int numElements() throws Exception {
    
    if (m_Classifier instanceof PartitionGenerator) {
      int size = 0;
      for (int i = 0; i < m_Classifiers.length; i++) {
        size += ((PartitionGenerator)m_Classifiers[i]).numElements();
      }
      return size;
    } else throw new Exception("Classifier: " + getClassifierSpec()
                               + " cannot generate a partition");
  }

  /**
   * Returns the revision string.
   * 
   * @return		the revision
   */
  public String getRevision() {
    return RevisionUtils.extract("$Revision: 15801 $");
  }

  /**
   * Main method for testing this class.
   *
   * @param argv the options
   */
  public static void main(String [] argv) {
    runClassifier(new RandomCommittee(), argv);
  }
}

