import ch.bfh.lpdg.DependencyScanner;
import ch.bfh.lpdg.InteractionHandler;
import ch.bfh.lpdg.datastructure.Dependency;
import ch.bfh.lpdg.datastructure.DependencyType;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;

import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.*;

class DependencyScannerTest {

    DependencyScanner dependencyScanner = DependencyScanner.getInstance();
    private final String PATH = "src/test/resources/test.txt";
    private final String DIR_PATH = "/src/test/resources/";

    @BeforeEach
    public void setUp(){
        InteractionHandler instance = InteractionHandler.getInstance();
        instance.setPath(instance.getUserDirectory() + DIR_PATH);
    }

    @Test
    void findDependencies_checkDepenencyName() {
        final File file = new File(PATH);
        final String expectedDependencyName = "test";

        var dependency = dependencyScanner.findDependencies(file.getAbsolutePath(), DependencyType.FILE, "", Collections.emptyList(), 5);
        final String actualDependencyName = dependency.getName();

        assertEquals(actualDependencyName, expectedDependencyName);
    }

    @Test
    void findDependencies_checkReturnValues() throws IOException {
        final File file = new File(PATH);

        var dependency = dependencyScanner.findDependencies(file.getAbsolutePath(), DependencyType.FILE, "", Collections.emptyList(), 5);
        // Using iterator().next() to get name out of Set<K>
        var dependencies = dependency.getDependencyList();

        final var usePackageDependencies =
                resolveDependenciesForType(dependencies, DependencyType.USE_PACKAGE);
        final var requirePackageDependencies =
                resolveDependenciesForType(dependencies, DependencyType.REQUIRE_PACKAGE);
        final var inputDependencies =
                resolveDependenciesForType(dependencies, DependencyType.INPUT);
        final var includeDependencies =
                resolveDependenciesForType(dependencies, DependencyType.INCLUDE);

        assertThat(usePackageDependencies).containsAll(List.of("xcolor", "makecell", "rexim"));
        assertThat(requirePackageDependencies).containsAll(List.of("testikus", "latexDev"));
        assertThat(inputDependencies).containsAll(List.of("smallFile"));
        assertThat(includeDependencies).containsAll(List.of("largeFile"));

        System.out.println(dependency);
    }

    @Test
    void test_ScanDepth() throws IOException {
        final File file = new File(PATH);

        //If we scan with the depth of 0, no packages should be found.
        var dependency = dependencyScanner.findDependencies(file.getAbsolutePath(), DependencyType.FILE, "", Collections.emptyList(), 0);
        var dependencies = dependency.getDependencyList();
        assertTrue(dependencies.size() == 0);


        //If we scan with the depth of 1, 7 packages should be found.
        dependency = dependencyScanner.findDependencies(file.getAbsolutePath(), DependencyType.FILE, "", Collections.emptyList(), 1);
        dependencies = dependency.getDependencyList();

        assertTrue(dependencies.size() == 7);
        //7 Packages, but these packages should not have been scanned again, since the depth was only 1
        boolean atLeastOneMoreFound = false;
        for (Dependency p : dependencies)
        {
            if(p.getDependencyList().size() > 0)
                atLeastOneMoreFound = true;
        }
        assertFalse(atLeastOneMoreFound);

        //If we scan with the depth of 2, we should again find 7 packages, but these packages should have dependencies themselves.
        dependency = dependencyScanner.findDependencies(file.getAbsolutePath(), DependencyType.FILE, "", Collections.emptyList(), 2);
        dependencies = dependency.getDependencyList();

        assertTrue(dependencies.size() == 7);
        //At least some packages should have dependencies themselves.
        for (Dependency p : dependencies)
        {
            if(p.getDependencyList().size() > 0)
                atLeastOneMoreFound = true;
        }
        assertTrue(atLeastOneMoreFound);
    }

    private List<String> resolveDependenciesForType(final List<Dependency> dependencyList, final DependencyType type) {
        return dependencyList.stream()
                .filter(d -> d.getType().equals(type)).toList()
                .stream()
                .map(Dependency::getName).collect(toList());
    }
}